// StandardJS, should-style assertions
// GUI tests for Documentation page functionality
// Generated by jinks - DO NOT EDIT manually

describe('TEI-Publisher Documentation Page', () => {
  beforeEach(() => {
    cy.viewport(1280, 720)
    // Universal intercepts (loginStub, timelineStub) are automatically set up in support/e2e.js
    cy.visit('/doc/documentation.xml')
  })

  describe('Documentation Page Load', () => {
    it('loads the documentation page successfully', () => {
      cy.url().should('include', '/doc/documentation.xml')
      cy.get('body').should('be.visible')
    })

    it('has valid page structure', () => {
      // Check main components exist
      cy.get('main').should('be.visible')
      cy.get('pb-document[id*="document1"]').should('exist')
      cy.get('pb-view[src*="document1"]').should('exist')
      
      // Check ToC is present
      cy.get('pb-load.toc').should('exist')
      
      // Check main content has loaded
      cy.get('main').should('be.visible')
      
      // Right panel should be hidden
      cy.get('aside.after.hidden-mobile').should('have.css', 'display', 'none')
    })

    it('shows main content area with actual content', () => {
      cy.get('main').should('be.visible')
      cy.get('pb-document[id*="document1"]').should('have.attr', 'path', 'doc/documentation.xml')
      cy.get('pb-view[src*="document1"]', { timeout: 15000 }).should('exist')
      
      // Wait for Introduction content to load - this is the actual documentation text
      cy.contains('Introduction', { timeout: 20000 }).should('be.visible')
      cy.get('main').should('not.contain', 'The request to the server failed')
      cy.get('main').should('not.contain', 'undefined')
    })

    it('displays documentation content', () => {
      // Wait for documentation content to load
      cy.get('pb-view').should('exist')
      // Accept either document1 or content-document1 as both are valid
      cy.get('pb-view').should('have.attr', 'src').and('match', /^(document1|content-document1)$/)
      
      // Wait for actual documentation content to appear
      cy.get('main', { timeout: 10000 }).should('be.visible')
      
      // Check that documentation content is loaded
      cy.get('body').then(($body) => {
        const text = $body.text()
        // Documentation should contain some text content
        expect(text.length).to.be.greaterThan(100)
      })
    })
  })

  describe('Documentation Navigation', () => {
    it('provides navigation elements', () => {
      cy.get('nav').should('be.visible')
    })

    it('has breadcrumb navigation when present', () => {
      cy.get('body').then($body => {
        if ($body.find('pb-view.breadcrumbs').length > 0) {
          cy.get('pb-view.breadcrumbs').should('be.visible')
        }
      })
    })

    it('displays table of contents', () => {
      cy.get('aside.before').should('be.visible')
      cy.get('h5').contains('Contents').should('be.visible')
      cy.get('pb-load.toc').should('exist')
    })

    it('table of contents links are clickable and trigger navigation', () => {
      cy.get('pb-load.toc').should('exist')
      cy.get('pb-link[node-id]', { timeout: 15000 }).should('exist')
      
      // Find first link with a hash or non-root node-id
      cy.get('pb-link[node-id]').first().then(($link) => {
        const hasHash = $link.attr('hash')
        const nodeId = $link.attr('node-id')
        
        // Skip if it's the root node
        if (!hasHash && nodeId === '3.6') {
          cy.get('pb-link[node-id]').eq(1).click({ force: true })
        } else {
          cy.wrap($link).click({ force: true })
        }
      })
      
      // Verify navigation occurred
      cy.get('main').should('be.visible')
    })

    it('table of contents has actual content', () => {
      cy.get('pb-load.toc').should('exist')
      cy.get('pb-load.toc', { timeout: 15000 }).should('not.be.empty')
      
      // Verify ToC has links loaded - pb-link elements are loaded dynamically
      // Use document-level query or wait for them to appear within pb-load.toc
      cy.get('pb-load.toc pb-link', { timeout: 15000 }).should('have.length.at.least', 1)
    })

    it('shows navigation controls', () => {
      cy.get('pb-navigation').should('have.length.at.least', 2)
    })
  })

  describe('Documentation Toolbar', () => {
    it('displays zoom controls', () => {
      cy.get('pb-zoom').should('have.length.at.least', 2)
    })

    it('shows navigation arrows', () => {
      cy.get('pb-navigation[direction="backward"]').should('exist')
      cy.get('pb-navigation[direction="forward"]').should('exist')
    })

    it('provides download options', () => {
      cy.get('pb-download').should('have.length.at.least', 1)
    })

    it('shows admin controls for authenticated users', () => {
      cy.get('pb-restricted').should('exist')
    })
  })

  describe('Documentation Sidebar', () => {
    it('hides timeline component on documentation page', () => {
      cy.get('pb-timeline').should('not.exist')
    })

    it('hides map component on documentation page', () => {
      cy.get('pb-leaflet-map').should('not.exist')
    })

    it('hides register view on documentation page', () => {
      cy.get('pb-register-view').should('not.exist')
    })
    
    it('right panel is hidden', () => {
      cy.get('aside.after.hidden-mobile').should('have.css', 'display', 'none')
    })
  })

  describe('Documentation Download Features', () => {
    it('provides PDF download options', () => {
      cy.get('pb-download[type="pdf"]').should('exist')
    })

    it('provides EPUB download option', () => {
      cy.get('pb-download[type="epub"]').should('exist')
    })

    it('provides XML view option', () => {
      cy.get('pb-edit-xml').should('exist')
    })

    it('provides markdown preview link', () => {
      cy.get('a[href*="markdown"]').should('exist')
    })
  })

  describe('Documentation Responsive Design', () => {
    it('adapts to mobile viewport', () => {
      cy.viewport(375, 667) // iPhone SE
      
      // Reload page to ensure layout adjusts to new viewport
      cy.reload()
      
      // Wait for page to stabilize
      cy.get('body').should('be.visible')
      
      // Wait for documentation content to load (check for pb-view which loads the content)
      cy.get('pb-view').should('exist')
      
      // On mobile, main content should be visible and ToC should be hidden by default
      cy.get('main').should('be.visible')
      
      // ToC should be hidden on mobile (has hidden-mobile class)
      // Note: This may expose a bug if ToC is visible when it shouldn't be
      cy.get('aside.before.hidden-mobile').should('have.css', 'display', 'none')
      
      // Mobile navigation should be available
      cy.get('.mobile.menubar').should('exist')
    })

    it('adapts to tablet viewport', () => {
      cy.viewport(768, 1024) // iPad
      
      // Reload page to ensure layout adjusts to new viewport
      cy.reload()
      
      // Wait for page to stabilize
      cy.get('body').should('be.visible')
      
      // Wait for documentation content to load (check for pb-view which loads the content)
      cy.get('pb-view').should('exist')
      
      // Wait for main to become visible - layout should adjust once content is loaded
      cy.get('main').should('be.visible')
      
      // Check that page structure is responsive
      cy.get('nav').should('be.visible')
      cy.get('.before-top').should('exist') // Navigation elements should exist
    })
  })

  describe('Documentation Accessibility', () => {
    it('has proper heading structure', () => {
      cy.get('h1, h2, h3, h4, h5, h6').should('have.length.at.least', 1)
    })

    it('has accessible navigation', () => {
      cy.get('nav').should('be.visible')
      cy.get('nav a').should('have.length.at.least', 1)
    })

    it('provides proper ARIA labels', () => {
      cy.get('[aria-label]').should('have.length.at.least', 1)
    })

    it('has accessible buttons', () => {
      cy.get('button').should('have.length.at.least', 1)
    })
  })

  describe('Documentation Error Handling', () => {
    it('handles documentation loading errors gracefully', () => {
      // Should handle any documentation-specific errors
      cy.get('body').should('be.visible')
    })

    it('displays loading states appropriately', () => {
      cy.get('pb-progress').should('exist')
    })
  })

  describe('Documentation Interactive Features', () => {
    it('supports keyboard navigation', () => {
      cy.get('pb-navigation[keyboard="left"]').should('exist')
      cy.get('pb-navigation[keyboard="right"]').should('exist')
    })

    it('provides zoom functionality', () => {
      cy.get('pb-zoom[direction="in"]').should('exist')
      cy.get('pb-zoom[direction="out"]').should('exist')
    })

    it('documentation page does not show timeline', () => {
      cy.get('pb-timeline').should('not.exist')
    })
  })

  describe('Documentation Content Structure', () => {
    it('displays document metadata', () => {
      cy.get('pb-document').should('exist')
      cy.get('pb-document').should('have.attr', 'path', 'doc/documentation.xml')
    })

    it('shows proper document view', () => {
      cy.get('pb-view').should('exist')
      cy.get('pb-view').should(($el) => {
        expect($el.attr('src')).to.include('document1')
      })
    })
  })
})

