(function polyfill() {
  const relList = document.createElement("link").relList;
  if (relList && relList.supports && relList.supports("modulepreload")) {
    return;
  }
  for (const link of document.querySelectorAll('link[rel="modulepreload"]')) {
    processPreload(link);
  }
  new MutationObserver((mutations) => {
    for (const mutation of mutations) {
      if (mutation.type !== "childList") {
        continue;
      }
      for (const node of mutation.addedNodes) {
        if (node.tagName === "LINK" && node.rel === "modulepreload")
          processPreload(node);
      }
    }
  }).observe(document, { childList: true, subtree: true });
  function getFetchOpts(link) {
    const fetchOpts = {};
    if (link.integrity) fetchOpts.integrity = link.integrity;
    if (link.referrerPolicy) fetchOpts.referrerPolicy = link.referrerPolicy;
    if (link.crossOrigin === "use-credentials")
      fetchOpts.credentials = "include";
    else if (link.crossOrigin === "anonymous") fetchOpts.credentials = "omit";
    else fetchOpts.credentials = "same-origin";
    return fetchOpts;
  }
  function processPreload(link) {
    if (link.ep)
      return;
    link.ep = true;
    const fetchOpts = getFetchOpts(link);
    fetch(link.href, fetchOpts);
  }
})();
const iconsSvg = '<!-- SPDX-License-Identifier: GPL-3.0-or-later -->\n<!-- SPDX-FileCopyrightText: 2025 e-editiones.org -->\n<svg xmlns="http://www.w3.org/2000/svg" style="display: none;">\n    <symbol id="icon-upload" fill="currentColor" viewBox="0 0 16 16">\n        <path\n            d="M.5 9.9a.5.5 0 0 1 .5.5v2.5a1 1 0 0 0 1 1h12a1 1 0 0 0 1-1v-2.5a.5.5 0 0 1 1 0v2.5a2 2 0 0 1-2 2H2a2 2 0 0 1-2-2v-2.5a.5.5 0 0 1 .5-.5" />\n        <path\n            d="M7.646 1.146a.5.5 0 0 1 .708 0l3 3a.5.5 0 0 1-.708.708L8.5 2.707V11.5a.5.5 0 0 1-1 0V2.707L5.354 4.854a.5.5 0 1 1-.708-.708z" />\n    </symbol>\n\n    <symbol id="icon-clipboard" fill="currentColor" viewBox="0 0 16 16">\n        <path\n            d="M4 1.5H3a2 2 0 0 0-2 2V14a2 2 0 0 0 2 2h10a2 2 0 0 0 2-2V3.5a2 2 0 0 0-2-2h-1v1h1a1 1 0 0 1 1 1V14a1 1 0 0 1-1 1H3a1 1 0 0 1-1-1V3.5a1 1 0 0 1 1-1h1z" />\n        <path\n            d="M9.5 1a.5.5 0 0 1 .5.5v1a.5.5 0 0 1-.5.5h-3a.5.5 0 0 1-.5-.5v-1a.5.5 0 0 1 .5-.5zm-3-1A1.5 1.5 0 0 0 5 1.5v1A1.5 1.5 0 0 0 6.5 4h3A1.5 1.5 0 0 0 11 2.5v-1A1.5 1.5 0 0 0 9.5 0z" />\n    </symbol>\n\n    <symbol id="icon-home" viewBox="0 0 16 16">\n        <path\n            d="M8.707 1.5a1 1 0 0 0-1.414 0L.646 8.146a.5.5 0 0 0 .708.708L2 8.207V13.5A1.5 1.5 0 0 0 3.5 15h9a1.5 1.5 0 0 0 1.5-1.5V8.207l.646.647a.5.5 0 0 0 .708-.708L13 5.793V2.5a.5.5 0 0 0-.5-.5h-1a.5.5 0 0 0-.5.5v1.293zM13 7.207V13.5a.5.5 0 0 1-.5.5h-9a.5.5 0 0 1-.5-.5V7.207l5-5z" />\n    </symbol>\n\n    <symbol id="icon-folder" viewBox="0 0 16 16">\n        <path\n            d="M.54 3.87.5 3a2 2 0 0 1 2-2h3.672a2 2 0 0 1 1.414.586l.828.828A2 2 0 0 0 9.828 3h3.982a2 2 0 0 1 1.992 2.181l-.637 7A2 2 0 0 1 13.174 14H2.826a2 2 0 0 1-1.991-1.819l-.637-7a2 2 0 0 1 .342-1.31zM2.19 4a1 1 0 0 0-.996 1.09l.637 7a1 1 0 0 0 .995.91h10.348a1 1 0 0 0 .995-.91l.637-7A1 1 0 0 0 13.81 4zm4.69-1.707A1 1 0 0 0 6.172 2H2.5a1 1 0 0 0-1 .981l.006.139q.323-.119.684-.12h5.396z" />\n    </symbol>\n\n    <symbol id="icon-folder-plus" viewBox="0 0 16 16">\n        <path\n            d="m.5 3 .04.87a2 2 0 0 0-.342 1.311l.637 7A2 2 0 0 0 2.826 14H9v-1H2.826a1 1 0 0 1-.995-.91l-.637-7A1 1 0 0 1 2.19 4h11.62a1 1 0 0 1 .996 1.09L14.54 8h1.005l.256-2.819A2 2 0 0 0 13.81 3H9.828a2 2 0 0 1-1.414-.586l-.828-.828A2 2 0 0 0 6.172 1H2.5a2 2 0 0 0-2 2m5.672-1a1 1 0 0 1 .707.293L7.586 3H2.19q-.362.002-.683.12L1.5 2.98a1 1 0 0 1 1-.98z" />\n        <path\n            d="M13.5 9a.5.5 0 0 1 .5.5V11h1.5a.5.5 0 1 1 0 1H14v1.5a.5.5 0 1 1-1 0V12h-1.5a.5.5 0 0 1 0-1H13V9.5a.5.5 0 0 1 .5-.5" />\n    </symbol>\n\n    <symbol id="icon-file" viewBox="0 0 16 16">\n        <path\n            d="M14 4.5V14a2 2 0 0 1-2 2H4a2 2 0 0 1-2-2V2a2 2 0 0 1 2-2h5.5zm-3 0A1.5 1.5 0 0 1 9.5 3V1H4a1 1 0 0 0-1 1v12a1 1 0 0 0 1 1h8a1 1 0 0 0 1-1V4.5z" />\n    </symbol>\n\n    <symbol id="icon-filetype-xml" viewBox="0 0 16 16">\n        <path fill-rule="evenodd"\n            d="M14 4.5V14a2 2 0 0 1-2 2v-1a1 1 0 0 0 1-1V4.5h-2A1.5 1.5 0 0 1 9.5 3V1H4a1 1 0 0 0-1 1v9H2V2a2 2 0 0 1 2-2h5.5zM3.527 11.85h-.893l-.823 1.439h-.036L.943 11.85H.012l1.227 1.983L0 15.85h.861l.853-1.415h.035l.85 1.415h.908l-1.254-1.992zm.954 3.999v-2.66h.038l.952 2.159h.516l.946-2.16h.038v2.661h.715V11.85h-.8l-1.14 2.596h-.025L4.58 11.85h-.806v3.999zm4.71-.674h1.696v.674H8.4V11.85h.791z" />\n    </symbol>\n    <symbol id="icon-filetype-json" viewBox="0 0 16 16">\n        <path fill-rule="evenodd"\n            d="M14 4.5V11h-1V4.5h-2A1.5 1.5 0 0 1 9.5 3V1H4a1 1 0 0 0-1 1v9H2V2a2 2 0 0 1 2-2h5.5zM4.151 15.29a1.2 1.2 0 0 1-.111-.449h.764a.58.58 0 0 0 .255.384q.105.073.25.114.142.041.319.041.245 0 .413-.07a.56.56 0 0 0 .255-.193.5.5 0 0 0 .084-.29.39.39 0 0 0-.152-.326q-.152-.12-.463-.193l-.618-.143a1.7 1.7 0 0 1-.539-.214 1 1 0 0 1-.352-.367 1.1 1.1 0 0 1-.123-.524q0-.366.19-.639.192-.272.528-.422.337-.15.777-.149.456 0 .779.152.326.153.5.41.18.255.2.566h-.75a.56.56 0 0 0-.12-.258.6.6 0 0 0-.246-.181.9.9 0 0 0-.37-.068q-.324 0-.512.152a.47.47 0 0 0-.185.384q0 .18.144.3a1 1 0 0 0 .404.175l.621.143q.326.075.566.211a1 1 0 0 1 .375.358q.135.222.135.56 0 .37-.188.656a1.2 1.2 0 0 1-.539.439q-.351.158-.858.158-.381 0-.665-.09a1.4 1.4 0 0 1-.478-.252 1.1 1.1 0 0 1-.29-.375m-3.104-.033a1.3 1.3 0 0 1-.082-.466h.764a.6.6 0 0 0 .074.27.5.5 0 0 0 .454.246q.285 0 .422-.164.137-.165.137-.466v-2.745h.791v2.725q0 .66-.357 1.005-.355.345-.985.345a1.6 1.6 0 0 1-.568-.094 1.15 1.15 0 0 1-.407-.266 1.1 1.1 0 0 1-.243-.39m9.091-1.585v.522q0 .384-.117.641a.86.86 0 0 1-.322.387.9.9 0 0 1-.47.126.9.9 0 0 1-.47-.126.87.87 0 0 1-.32-.387 1.55 1.55 0 0 1-.117-.641v-.522q0-.386.117-.641a.87.87 0 0 1 .32-.387.87.87 0 0 1 .47-.129q.265 0 .47.129a.86.86 0 0 1 .322.387q.117.255.117.641m.803.519v-.513q0-.565-.205-.973a1.46 1.46 0 0 0-.59-.63q-.38-.22-.916-.22-.534 0-.92.22a1.44 1.44 0 0 0-.589.628q-.205.407-.205.975v.513q0 .562.205.973.205.407.589.626.386.217.92.217.536 0 .917-.217.384-.22.589-.626.204-.41.205-.973m1.29-.935v2.675h-.746v-3.999h.662l1.752 2.66h.032v-2.66h.75v4h-.656l-1.761-2.676z" />\n    </symbol>\n    <symbol id="icon-filetype-css" viewBox="0 0 16 16">\n        <path fill-rule="evenodd"\n            d="M14 4.5V14a2 2 0 0 1-2 2h-1v-1h1a1 1 0 0 0 1-1V4.5h-2A1.5 1.5 0 0 1 9.5 3V1H4a1 1 0 0 0-1 1v9H2V2a2 2 0 0 1 2-2h5.5zM3.397 14.841a1.13 1.13 0 0 0 .401.823q.195.162.478.252.284.091.665.091.507 0 .859-.158.354-.158.539-.44.187-.284.187-.656 0-.336-.134-.56a1 1 0 0 0-.375-.357 2 2 0 0 0-.566-.21l-.621-.144a1 1 0 0 1-.404-.176.37.37 0 0 1-.144-.299q0-.234.185-.384.188-.152.512-.152.214 0 .37.068a.6.6 0 0 1 .246.181.56.56 0 0 1 .12.258h.75a1.1 1.1 0 0 0-.2-.566 1.2 1.2 0 0 0-.5-.41 1.8 1.8 0 0 0-.78-.152q-.439 0-.776.15-.337.149-.527.421-.19.273-.19.639 0 .302.122.524.124.223.352.367.228.143.539.213l.618.144q.31.073.463.193a.39.39 0 0 1 .152.326.5.5 0 0 1-.085.29.56.56 0 0 1-.255.193q-.167.07-.413.07-.175 0-.32-.04a.8.8 0 0 1-.248-.115.58.58 0 0 1-.255-.384zM.806 13.693q0-.373.102-.633a.87.87 0 0 1 .302-.399.8.8 0 0 1 .475-.137q.225 0 .398.097a.7.7 0 0 1 .272.26.85.85 0 0 1 .12.381h.765v-.072a1.33 1.33 0 0 0-.466-.964 1.4 1.4 0 0 0-.489-.272 1.8 1.8 0 0 0-.606-.097q-.534 0-.911.223-.375.222-.572.632-.195.41-.196.979v.498q0 .568.193.976.197.407.572.626.375.217.914.217.439 0 .785-.164t.55-.454a1.27 1.27 0 0 0 .226-.674v-.076h-.764a.8.8 0 0 1-.118.363.7.7 0 0 1-.272.25.9.9 0 0 1-.401.087.85.85 0 0 1-.478-.132.83.83 0 0 1-.299-.392 1.7 1.7 0 0 1-.102-.627zM6.78 15.29a1.2 1.2 0 0 1-.111-.449h.764a.58.58 0 0 0 .255.384q.106.073.25.114.142.041.319.041.245 0 .413-.07a.56.56 0 0 0 .255-.193.5.5 0 0 0 .085-.29.39.39 0 0 0-.153-.326q-.152-.12-.463-.193l-.618-.143a1.7 1.7 0 0 1-.539-.214 1 1 0 0 1-.351-.367 1.1 1.1 0 0 1-.123-.524q0-.366.19-.639.19-.272.527-.422t.777-.149q.456 0 .779.152.326.153.5.41.18.255.2.566h-.75a.56.56 0 0 0-.12-.258.6.6 0 0 0-.246-.181.9.9 0 0 0-.37-.068q-.324 0-.512.152a.47.47 0 0 0-.184.384q0 .18.143.3a1 1 0 0 0 .404.175l.621.143q.326.075.566.211t.375.358.135.56q0 .37-.188.656a1.2 1.2 0 0 1-.539.439q-.351.158-.858.158-.381 0-.665-.09a1.4 1.4 0 0 1-.478-.252 1.1 1.1 0 0 1-.29-.375" />\n    </symbol>\n    <symbol id="icon-filetype-javascript" viewBox="0 0 16 16">\n        <path fill-rule="evenodd"\n            d="M14 4.5V14a2 2 0 0 1-2 2H8v-1h4a1 1 0 0 0 1-1V4.5h-2A1.5 1.5 0 0 1 9.5 3V1H4a1 1 0 0 0-1 1v9H2V2a2 2 0 0 1 2-2h5.5zM3.186 15.29a1.2 1.2 0 0 1-.111-.449h.765a.58.58 0 0 0 .255.384q.105.073.249.114.143.041.319.041.246 0 .413-.07a.56.56 0 0 0 .255-.193.5.5 0 0 0 .085-.29.39.39 0 0 0-.153-.326q-.151-.12-.462-.193l-.619-.143a1.7 1.7 0 0 1-.539-.214 1 1 0 0 1-.351-.367 1.1 1.1 0 0 1-.123-.524q0-.366.19-.639.19-.272.528-.422.336-.15.776-.149.457 0 .78.152.324.153.5.41.18.255.2.566h-.75a.56.56 0 0 0-.12-.258.6.6 0 0 0-.247-.181.9.9 0 0 0-.369-.068q-.325 0-.513.152a.47.47 0 0 0-.184.384q0 .18.143.3a1 1 0 0 0 .405.175l.62.143q.327.075.566.211.24.136.375.358t.135.56q0 .37-.188.656a1.2 1.2 0 0 1-.539.439q-.351.158-.858.158-.381 0-.665-.09a1.4 1.4 0 0 1-.478-.252 1.1 1.1 0 0 1-.29-.375m-3.104-.033A1.3 1.3 0 0 1 0 14.791h.765a.6.6 0 0 0 .073.27.5.5 0 0 0 .454.246q.285 0 .422-.164.138-.165.138-.466v-2.745h.79v2.725q0 .66-.357 1.005-.354.345-.984.345a1.6 1.6 0 0 1-.569-.094 1.15 1.15 0 0 1-.407-.266 1.1 1.1 0 0 1-.243-.39" />\n    </symbol>\n    <symbol id="icon-filetype-html" viewBox="0 0 16 16">\n        <path fill-rule="evenodd"\n            d="M14 4.5V11h-1V4.5h-2A1.5 1.5 0 0 1 9.5 3V1H4a1 1 0 0 0-1 1v9H2V2a2 2 0 0 1 2-2h5.5zm-9.736 7.35v3.999h-.791v-1.714H1.79v1.714H1V11.85h.791v1.626h1.682V11.85h.79Zm2.251.662v3.337h-.794v-3.337H4.588v-.662h3.064v.662zm2.176 3.337v-2.66h.038l.952 2.159h.516l.946-2.16h.038v2.661h.715V11.85h-.8l-1.14 2.596H9.93L8.79 11.85h-.805v3.999zm4.71-.674h1.696v.674H12.61V11.85h.79v3.325Z" />\n    </symbol>\n    <symbol id="icon-filetype-xquery" viewBox="0 0 16 16">\n        <path\n            d="M14 4.5V14a2 2 0 0 1-2 2H4a2 2 0 0 1-2-2V2a2 2 0 0 1 2-2h5.5zm-3 0A1.5 1.5 0 0 1 9.5 3V1H4a1 1 0 0 0-1 1v12a1 1 0 0 0 1 1h8a1 1 0 0 0 1-1V4.5z" />\n        <path\n            d="M8.646 6.646a.5.5 0 0 1 .708 0l2 2a.5.5 0 0 1 0 .708l-2 2a.5.5 0 0 1-.708-.708L10.293 9 8.646 7.354a.5.5 0 0 1 0-.708m-1.292 0a.5.5 0 0 0-.708 0l-2 2a.5.5 0 0 0 0 .708l2 2a.5.5 0 0 0 .708-.708L5.707 9l1.647-1.646a.5.5 0 0 0 0-.708" />\n    </symbol>\n    <symbol id="icon-rename" viewBox="0 0 16 16">\n        <path\n            d="M15.502 1.94a.5.5 0 0 1 0 .706L14.459 3.69l-2-2L13.502.646a.5.5 0 0 1 .707 0l1.293 1.293zm-1.75 2.456-2-2L4.939 9.21a.5.5 0 0 0-.121.196l-.805 2.414a.25.25 0 0 0 .316.316l2.414-.805a.5.5 0 0 0 .196-.12l6.813-6.814z" />\n        <path fill-rule="evenodd"\n            d="M1 13.5A1.5 1.5 0 0 0 2.5 15h11a1.5 1.5 0 0 0 1.5-1.5v-6a.5.5 0 0 0-1 0v6a.5.5 0 0 1-.5.5h-11a.5.5 0 0 1-.5-.5v-11a.5.5 0 0 1 .5-.5H9a.5.5 0 0 0 0-1H2.5A1.5 1.5 0 0 0 1 2.5z" />\n    </symbol>\n\n    <symbol id="icon-delete" viewBox="0 0 16 16">\n        <path\n            d="M6.5 1h3a.5.5 0 0 1 .5.5v1H6v-1a.5.5 0 0 1 .5-.5M11 2.5v-1A1.5 1.5 0 0 0 9.5 0h-3A1.5 1.5 0 0 0 5 1.5v1H1.5a.5.5 0 0 0 0 1h.538l.853 10.66A2 2 0 0 0 4.885 16h6.23a2 2 0 0 0 1.994-1.84l.853-10.66h.538a.5.5 0 0 0 0-1zm1.958 1-.846 10.58a1 1 0 0 1-.997.92h-6.23a1 1 0 0 1-.997-.92L3.042 3.5zm-7.487 1a.5.5 0 0 1 .528.47l.5 8.5a.5.5 0 0 1-.998.06L5 5.03a.5.5 0 0 1 .47-.53Zm5.058 0a.5.5 0 0 1 .47.53l-.5 8.5a.5.5 0 1 1-.998-.06l.5-8.5a.5.5 0 0 1 .528-.47M8 4.5a.5.5 0 0 1 .5.5v8.5a.5.5 0 0 1-1 0V5a.5.5 0 0 1 .5-.5" />\n    </symbol>\n\n    <symbol id="icon-copy" viewBox="0 0 16 16">\n        <path fill-rule="evenodd"\n            d="M4 2a2 2 0 0 1 2-2h8a2 2 0 0 1 2 2v8a2 2 0 0 1-2 2H6a2 2 0 0 1-2-2zm2-1a1 1 0 0 0-1 1v8a1 1 0 0 0 1 1h8a1 1 0 0 0 1-1V2a1 1 0 0 0-1-1zM2 5a1 1 0 0 0-1 1v8a1 1 0 0 0 1 1h8a1 1 0 0 0 1-1v-1h1v1a2 2 0 0 1-2 2H2a2 2 0 0 1-2-2V6a2 2 0 0 1 2-2h1v1z" />\n    </symbol>\n\n    <symbol id="icon-copy-path" viewBox="0 0 16 16">\n        <path fill-rule="evenodd"\n            d="M8 7a.5.5 0 0 1 .5.5V9H10a.5.5 0 0 1 0 1H8.5v1.5a.5.5 0 0 1-1 0V10H6a.5.5 0 0 1 0-1h1.5V7.5A.5.5 0 0 1 8 7" />\n        <path\n            d="M4 1.5H3a2 2 0 0 0-2 2V14a2 2 0 0 0 2 2h10a2 2 0 0 0 2-2V3.5a2 2 0 0 0-2-2h-1v1h1a1 1 0 0 1 1 1V14a1 1 0 0 1-1 1H3a1 1 0 0 1-1-1V3.5a1 1 0 0 1 1-1h1z" />\n        <path\n            d="M9.5 1a.5.5 0 0 1 .5.5v1a.5.5 0 0 1-.5.5h-3a.5.5 0 0 1-.5-.5v-1a.5.5 0 0 1 .5-.5zm-3-1A1.5 1.5 0 0 0 5 1.5v1A1.5 1.5 0 0 0 6.5 4h3A1.5 1.5 0 0 0 11 2.5v-1A1.5 1.5 0 0 0 9.5 0z" />\n    </symbol>\n\n    <symbol id="icon-download" viewBox="0 0 16 16">\n        <path\n            d="M.5 9.9a.5.5 0 0 1 .5.5v2.5a1 1 0 0 0 1 1h12a1 1 0 0 0 1-1v-2.5a.5.5 0 0 1 1 0v2.5a2 2 0 0 1-2 2H2a2 2 0 0 1-2-2v-2.5a.5.5 0 0 1 .5-.5" />\n        <path\n            d="M7.646 11.854a.5.5 0 0 0 .708 0l3-3a.5.5 0 0 0-.708-.708L8.5 10.293V1.5a.5.5 0 0 0-1 0v8.793L5.354 8.146a.5.5 0 1 0-.708.708z" />\n    </symbol>\n\n    <symbol id="icon-cut" viewBox="0 0 16 16">\n        <path\n            d="M3.5 3.5c-.614-.884-.074-1.962.858-2.5L8 7.226 11.642 1c.932.538 1.472 1.616.858 2.5L8.81 8.61l1.556 2.661a2.5 2.5 0 1 1-.794.637L8 9.73l-1.572 2.177a2.5 2.5 0 1 1-.794-.637L7.19 8.61zm2.5 10a1.5 1.5 0 1 0-3 0 1.5 1.5 0 0 0 3 0m7 0a1.5 1.5 0 1 0-3 0 1.5 1.5 0 0 0 3 0" />\n    </symbol>\n\n    <symbol id="icon-edit" viewBox="0 0 16 16">\n        <path\n            d="M15.502 1.94a.5.5 0 0 1 0 .706L14.459 3.69l-2-2L13.502.646a.5.5 0 0 1 .707 0l1.293 1.293zm-1.75 2.456-2-2L4.939 9.21a.5.5 0 0 0-.121.196l-.805 2.414a.25.25 0 0 0 .316.316l2.414-.805a.5.5 0 0 0 .196-.12l6.813-6.814z" />\n        <path fill-rule="evenodd"\n            d="M1 13.5A1.5 1.5 0 0 0 2.5 15h11a1.5 1.5 0 0 0 1.5-1.5v-6a.5.5 0 0 0-1 0v6a.5.5 0 0 1-.5.5h-11a.5.5 0 0 1-.5-.5v-11a.5.5 0 0 1 .5-.5H9a.5.5 0 0 0 0-1H2.5A1.5 1.5 0 0 0 1 2.5z" />\n    </symbol>\n</svg>\n';
const stylesCss = "/* SPDX-License-Identifier: GPL-3.0-or-later */\n/* SPDX-FileCopyrightText: 2025 e-editiones.org */\n\n:host {\n  display: block;\n  width: 100%;\n  height: 100%;\n  font-size: var(--jinks-file-manager-font-size);\n  color: var(--jinks-file-manager-color-text);\n  \n  /* Font size */\n  --jinks-file-manager-font-size: 1rem;\n  --jinks-file-manager-font-size-small: 0.875rem;\n  --jinks-file-manager-font-size-medium: 1.125rem;\n\n  /* Base color palette */\n  --jinks-file-manager-color-lightest: #fff;\n  --jinks-file-manager-color-light: #f5f5f5;\n  --jinks-file-manager-color-medium: #e0e0e0;\n  --jinks-file-manager-color-dark: #999;\n  --jinks-file-manager-color-darker: #666;\n  --jinks-file-manager-color-darkest: #333;\n  --jinks-file-manager-color-primary: #2196f3;\n  --jinks-file-manager-color-primary-dark: #1976d2;\n  --jinks-file-manager-color-primary-darker: #1565c0;\n  --jinks-file-manager-color-error: #f44336;\n  --jinks-file-manager-color-success: #4caf50;\n  --jinks-file-manager-color-folder: #ffa726;\n  \n  /* Derived colors - backgrounds (dynamically computed) */\n  --jinks-file-manager-bg-primary: var(--jinks-file-manager-color-lightest);\n  --jinks-file-manager-bg-secondary: color-mix(in srgb, var(--jinks-file-manager-color-lightest) 95%, var(--jinks-file-manager-color-light));\n  --jinks-file-manager-bg-hover: color-mix(in srgb, var(--jinks-file-manager-color-light) 80%, var(--jinks-file-manager-color-medium));\n  --jinks-file-manager-bg-active: color-mix(in srgb, var(--jinks-file-manager-color-medium) 70%, var(--jinks-file-manager-color-dark));\n  --jinks-file-manager-bg-selected: color-mix(in srgb, var(--jinks-file-manager-color-primary) 10%, transparent);\n  --jinks-file-manager-bg-error: color-mix(in srgb, var(--jinks-file-manager-color-error) 10%, transparent);\n  --jinks-file-manager-bg-success: color-mix(in srgb, var(--jinks-file-manager-color-success) 10%, transparent);\n  --jinks-file-manager-bg-scrollbar-track: color-mix(in srgb, var(--jinks-file-manager-color-lightest) 90%, var(--jinks-file-manager-color-light));\n  --jinks-file-manager-bg-scrollbar-thumb: color-mix(in srgb, var(--jinks-file-manager-color-medium) 50%, var(--jinks-file-manager-color-dark));\n  --jinks-file-manager-bg-scrollbar-thumb-hover: color-mix(in srgb, var(--jinks-file-manager-color-dark) 60%, var(--jinks-file-manager-color-darker));\n  \n  /* Derived colors - text (direct references) */\n  --jinks-file-manager-color-text: var(--jinks-file-manager-color-darkest);\n  --jinks-file-manager-color-text-secondary: var(--jinks-file-manager-color-darker);\n  --jinks-file-manager-color-text-muted: var(--jinks-file-manager-color-dark);\n  --jinks-file-manager-color-text-selected: var(--jinks-file-manager-color-primary-dark);\n  --jinks-file-manager-color-text-error: color-mix(in srgb, var(--jinks-file-manager-color-error) 60%, var(--jinks-file-manager-color-darkest));\n  --jinks-file-manager-color-text-success: color-mix(in srgb, var(--jinks-file-manager-color-success) 60%, var(--jinks-file-manager-color-darkest));\n  \n  /* Derived colors - borders (dynamically computed from base colors) */\n  --jinks-file-manager-border-default: var(--jinks-file-manager-color-medium);\n  --jinks-file-manager-border-button: color-mix(in srgb, var(--jinks-file-manager-color-medium) 80%, var(--jinks-file-manager-color-darkest));\n  --jinks-file-manager-border-hover: color-mix(in srgb, var(--jinks-file-manager-color-medium) 50%, var(--jinks-file-manager-color-dark));\n  --jinks-file-manager-border-selected: var(--jinks-file-manager-color-primary);\n  --jinks-file-manager-border-info: color-mix(in srgb, var(--jinks-file-manager-color-primary) 60%, var(--jinks-file-manager-color-lightest));\n  --jinks-file-manager-border-error: color-mix(in srgb, var(--jinks-file-manager-color-error) 80%, var(--jinks-file-manager-color-lightest));\n  --jinks-file-manager-border-success: color-mix(in srgb, var(--jinks-file-manager-color-success) 80%, var(--jinks-file-manager-color-lightest));\n  \n  /* Derived colors - button states (dynamically computed) */\n  --jinks-file-manager-color-error-hover: color-mix(in srgb, var(--jinks-file-manager-color-error) 50%, var(--jinks-file-manager-color-darkest));\n  --jinks-file-manager-color-success-hover: color-mix(in srgb, var(--jinks-file-manager-color-success) 50%, var(--jinks-file-manager-color-darkest));\n}\n\n.file-manager {\n  display: flex;\n  flex-direction: column;\n  height: 100%;\n  background: var(--jinks-file-manager-bg-primary);\n  border: 1px solid var(--jinks-file-manager-border-default);\n  border-radius: 8px;\n  overflow: hidden;\n}\n\n/* Toolbar */\n.toolbar {\n  display: flex;\n  align-items: center;\n  gap: 8px;\n  padding: 12px 16px;\n  background: var(--jinks-file-manager-bg-secondary);\n  border-bottom: 1px solid var(--jinks-file-manager-border-default);\n  flex-shrink: 0;\n}\n\n.toolbar button {\n  display: flex;\n  align-items: center;\n  gap: 6px;\n  padding: 8px 12px;\n  background: var(--jinks-file-manager-bg-primary);\n  border: 1px solid var(--jinks-file-manager-border-button);\n  border-radius: 4px;\n  cursor: pointer;\n  font-size: var(--jinks-file-manager-font-size-small);\n  color: var(--jinks-file-manager-color-text);\n  transition: all 0.2s;\n}\n\n.toolbar button:hover:not(:disabled) {\n  background: var(--jinks-file-manager-bg-hover);\n  border-color: var(--jinks-file-manager-border-hover);\n}\n\n.toolbar button:active:not(:disabled) {\n  background: var(--jinks-file-manager-bg-active);\n}\n\n.toolbar button:disabled {\n  opacity: 0.5;\n  cursor: not-allowed;\n}\n\n.toolbar button svg {\n  flex-shrink: 0;\n}\n\n.spacer {\n  flex: 1;\n}\n\n.clipboard-indicator {\n  display: flex;\n  align-items: center;\n  gap: 6px;\n  padding: 6px 12px;\n  background: var(--jinks-file-manager-bg-selected);\n  border: 1px solid var(--jinks-file-manager-border-info);\n  border-radius: 4px;\n  font-size: var(--jinks-file-manager-font-size-small);\n  color: var(--jinks-file-manager-color-text-selected);\n}\n\n.clipboard-indicator svg {\n  flex-shrink: 0;\n}\n\n/* Breadcrumb */\n.breadcrumb {\n  display: flex;\n  align-items: center;\n  gap: 4px;\n  padding: 8px 16px;\n  background: var(--jinks-file-manager-bg-secondary);\n  border-bottom: 1px solid var(--jinks-file-manager-border-default);\n  flex-shrink: 0;\n  overflow-x: auto;\n}\n\n.breadcrumb-item {\n  display: flex;\n  align-items: center;\n  padding: 4px 8px;\n  background: transparent;\n  border: none;\n  border-radius: 4px;\n  cursor: pointer;\n  font-size: var(--jinks-file-manager-font-size-small);\n  color: var(--jinks-file-manager-color-text-secondary);\n  transition: all 0.2s;\n  white-space: nowrap;\n}\n\n.breadcrumb-item:hover:not(.current) {\n  background: var(--jinks-file-manager-bg-hover);\n  color: var(--jinks-file-manager-color-text);\n}\n\n.breadcrumb-item.current {\n  color: var(--jinks-file-manager-color-text);\n  font-weight: 500;\n  cursor: default;\n}\n\n.breadcrumb-item:disabled {\n  cursor: default;\n  opacity: 1;\n  pointer-events: none;\n}\n\n.breadcrumb-item.home {\n  padding: 4px;\n}\n\n.breadcrumb-item.home svg {\n  width: 16px;\n  height: 16px;\n}\n\n.breadcrumb-separator {\n  color: var(--jinks-file-manager-color-text-muted);\n  user-select: none;\n}\n\n/* Content Area */\n.content {\n  flex: 1;\n  display: flex;\n  flex-direction: column;\n  overflow: hidden;\n  position: relative;\n}\n\n.content.drag-over {\n  background: rgba(33, 150, 243, 0.1);\n  border: 2px dashed var(--jinks-file-manager-color-primary);\n  border-radius: 8px;\n}\n\n.grid-container {\n  flex: 1;\n  display: grid;\n  grid-template-columns: repeat(auto-fill, minmax(120px, 1fr));\n  gap: 16px;\n  padding: 16px;\n  overflow-y: auto;\n  overflow-x: hidden;\n  align-content: start;\n  min-height: 0;\n}\n\n.grid-item {\n  display: flex;\n  flex-direction: column;\n  align-items: center;\n  padding: 12px;\n  border: 2px solid transparent;\n  border-radius: 8px;\n  cursor: pointer;\n  transition: all 0.2s;\n  background: var(--jinks-file-manager-bg-primary);\n  position: relative;\n}\n\n.grid-item:hover {\n  background: var(--jinks-file-manager-bg-secondary);\n  border-color: var(--jinks-file-manager-border-default);\n  transform: translateY(-2px);\n  box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);\n}\n\n.grid-item.selected {\n  background: var(--jinks-file-manager-bg-selected);\n  border-color: var(--jinks-file-manager-border-selected);\n}\n\n.grid-item.folder {\n  cursor: pointer;\n}\n\n.item-icon {\n  display: flex;\n  align-items: center;\n  justify-content: center;\n  width: 64px;\n  height: 64px;\n  margin-bottom: 8px;\n  color: var(--jinks-file-manager-color-text-secondary);\n  transition: color 0.2s;\n  position: relative;\n}\n\n.item-icon .thumbnail-image {\n  width: 100%;\n  height: 100%;\n  object-fit: contain;\n  border-radius: 4px;\n  background: var(--jinks-file-manager-color-gray-light);\n}\n\n.item-icon .thumbnail-fallback {\n  position: absolute;\n  top: 50%;\n  left: 50%;\n  transform: translate(-50%, -50%);\n}\n\n.grid-item.folder .item-icon {\n  color: var(--jinks-file-manager-color-folder);\n}\n\n.grid-item.selected .item-icon {\n  color: var(--jinks-file-manager-color-primary);\n}\n\n.item-name {\n  text-align: center;\n  font-size: var(--jinks-file-manager-font-size-small);\n  color: var(--jinks-file-manager-color-text);\n  word-break: break-word;\n  max-width: 100%;\n  overflow: hidden;\n  text-overflow: ellipsis;\n  display: -webkit-box;\n  -webkit-line-clamp: 2;\n  -webkit-box-orient: vertical;\n  line-height: 1.4;\n}\n\n.grid-item.selected .item-name {\n  color: var(--jinks-file-manager-color-text-selected);\n  font-weight: 500;\n}\n\n/* Loading */\n.loading {\n  display: flex;\n  flex-direction: column;\n  align-items: center;\n  justify-content: center;\n  gap: 12px;\n  padding: 40px;\n  color: var(--jinks-file-manager-color-text-secondary);\n}\n\n.spinner {\n  width: 32px;\n  height: 32px;\n  border: 3px solid var(--jinks-file-manager-color-gray-light);\n  border-top: 3px solid var(--jinks-file-manager-color-primary);\n  border-radius: 50%;\n  animation: spin 1s linear infinite;\n}\n\n@keyframes spin {\n  0% { transform: rotate(0deg); }\n  100% { transform: rotate(360deg); }\n}\n\n/* Load More */\n.load-more-container {\n  display: flex;\n  justify-content: center;\n  padding: 16px;\n  border-top: 1px solid var(--jinks-file-manager-border-default);\n}\n\n.btn-load-more {\n  padding: 10px 20px;\n  background: var(--jinks-file-manager-color-primary);\n  color: white;\n  border: none;\n  border-radius: 4px;\n  cursor: pointer;\n  font-size: var(--jinks-file-manager-font-size-small);\n  transition: background 0.2s;\n}\n\n.btn-load-more:hover {\n  background: var(--jinks-file-manager-color-primary-dark);\n}\n\n.btn-load-more:active {\n  background: var(--jinks-file-manager-color-primary-darker);\n}\n\n/* Empty State */\n.empty-state {\n  display: flex;\n  align-items: center;\n  justify-content: center;\n  padding: 60px 20px;\n  color: var(--jinks-file-manager-color-text-muted);\n  font-size: var(--jinks-file-manager-font-size-small);\n}\n\n/* Message Footer */\n.message-footer {\n  display: flex;\n  align-items: center;\n  justify-content: space-between;\n  padding: 12px 16px;\n  background: var(--jinks-file-manager-bg-selected);\n  border-top: 1px solid var(--jinks-file-manager-border-info);\n  color: var(--jinks-file-manager-color-text-selected);\n  font-size: var(--jinks-file-manager-font-size-small);\n  flex-shrink: 0;\n  animation: slideUp 0.3s ease-out;\n}\n\n.message-footer.message-error {\n  background: var(--jinks-file-manager-bg-error);\n  border-top-color: var(--jinks-file-manager-border-error);\n  color: var(--jinks-file-manager-color-text-error);\n}\n\n.message-footer.message-success {\n  background: var(--jinks-file-manager-bg-success);\n  border-top-color: var(--jinks-file-manager-border-success);\n  color: var(--jinks-file-manager-color-text-success);\n}\n\n.message-footer.message-info {\n  background: var(--jinks-file-manager-bg-selected);\n  border-top-color: var(--jinks-file-manager-border-info);\n  color: var(--jinks-file-manager-color-text-selected);\n}\n\n.message-text {\n  flex: 1;\n  margin-right: 12px;\n}\n\n.message-input {\n  flex: 1;\n  max-width: 300px;\n  padding: 6px 12px;\n  border: 1px solid currentColor;\n  background: white;\n  color: var(--jinks-file-manager-color-text);\n  border-radius: 4px;\n  font-size: var(--jinks-file-manager-font-size-small);\n  margin-right: 8px;\n  opacity: 0.9;\n  transition: opacity 0.2s;\n}\n\n.message-input:focus {\n  outline: none;\n  opacity: 1;\n  border-color: currentColor;\n  box-shadow: 0 0 0 2px rgba(0, 0, 0, 0.1);\n}\n\n.message-footer.message-error .message-input {\n  border-color: var(--jinks-file-manager-color-text-error);\n}\n\n.message-footer.message-info .message-input {\n  border-color: var(--jinks-file-manager-color-text-selected);\n}\n\n.message-footer.message-success .message-input {\n  border-color: var(--jinks-file-manager-color-text-success);\n}\n\n.message-actions {\n  display: flex;\n  align-items: center;\n  gap: 8px;\n}\n\n.message-confirm-btn,\n.message-cancel-btn {\n  padding: 6px 16px;\n  border: 1px solid currentColor;\n  background: transparent;\n  color: inherit;\n  border-radius: 4px;\n  cursor: pointer;\n  font-size: var(--jinks-file-manager-font-size-small);\n  font-weight: 500;\n  transition: all 0.2s;\n  opacity: 0.9;\n}\n\n.message-confirm-btn:hover,\n.message-cancel-btn:hover {\n  opacity: 1;\n  background: rgba(0, 0, 0, 0.05);\n}\n\n.message-confirm-btn:active,\n.message-cancel-btn:active {\n  background: rgba(0, 0, 0, 0.1);\n}\n\n.message-confirm-btn {\n  background: currentColor;\n  color: white;\n  border-color: currentColor;\n  opacity: 1;\n}\n\n.message-confirm-btn:hover {\n  filter: brightness(0.9);\n}\n\n/* Specific styling for error type confirm button */\n.message-footer.message-error .message-confirm-btn {\n  background: var(--jinks-file-manager-color-text-error);\n  color: white;\n  border-color: var(--jinks-file-manager-color-text-error);\n}\n\n.message-footer.message-error .message-confirm-btn:hover {\n  background: var(--jinks-file-manager-color-error-hover);\n}\n\n/* Specific styling for info type confirm button */\n.message-footer.message-info .message-confirm-btn {\n  background: var(--jinks-file-manager-color-text-selected);\n  color: white;\n  border-color: var(--jinks-file-manager-color-text-selected);\n}\n\n.message-footer.message-info .message-confirm-btn:hover {\n  background: var(--jinks-file-manager-color-primary-darker);\n}\n\n/* Specific styling for success type confirm button */\n.message-footer.message-success .message-confirm-btn {\n  background: var(--jinks-file-manager-color-text-success);\n  color: white;\n  border-color: var(--jinks-file-manager-color-text-success);\n}\n\n.message-footer.message-success .message-confirm-btn:hover {\n  background: var(--jinks-file-manager-color-success-hover);\n}\n\n.message-close {\n  background: transparent;\n  border: none;\n  color: inherit;\n  font-size: var(--jinks-file-manager-font-size-medium);\n  line-height: 1;\n  cursor: pointer;\n  padding: 0;\n  width: 24px;\n  height: 24px;\n  display: flex;\n  align-items: center;\n  justify-content: center;\n  border-radius: 4px;\n  transition: background 0.2s;\n  opacity: 0.7;\n}\n\n.message-close:hover {\n  background: rgba(0, 0, 0, 0.1);\n  opacity: 1;\n}\n\n.message-close:active {\n  background: rgba(0, 0, 0, 0.2);\n}\n\n@keyframes slideUp {\n  from {\n    transform: translateY(100%);\n    opacity: 0;\n  }\n  to {\n    transform: translateY(0);\n    opacity: 1;\n  }\n}\n\n/* Context Menu */\n.context-menu {\n  position: fixed;\n  background: white;\n  border: 1px solid var(--jinks-file-manager-border-button);\n  border-radius: 6px;\n  box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);\n  padding: 4px;\n  z-index: 10000;\n  min-width: 180px;\n  font-size: var(--jinks-file-manager-font-size-small);\n}\n\n.context-menu-item {\n  display: flex;\n  align-items: center;\n  gap: 8px;\n  padding: 8px 12px;\n  cursor: pointer;\n  border-radius: 4px;\n  color: var(--jinks-file-manager-color-text);\n  transition: background 0.15s;\n}\n\n.context-menu-item:hover {\n  background: var(--jinks-file-manager-bg-hover);\n}\n\n.context-menu-item svg {\n  flex-shrink: 0;\n  width: 16px;\n  height: 16px;\n  color: var(--jinks-file-manager-color-text-secondary);\n}\n\n.context-menu-separator {\n  height: 1px;\n  background: var(--jinks-file-manager-border-default);\n  margin: 4px 0;\n}\n\n/* Responsive */\n@media (max-width: 768px) {\n  .grid-container {\n    grid-template-columns: repeat(auto-fill, minmax(100px, 1fr));\n    gap: 12px;\n    padding: 12px;\n  }\n  \n  .item-icon {\n    width: 48px;\n    height: 48px;\n  }\n  \n  .toolbar {\n    padding: 8px 12px;\n  }\n  \n  .toolbar button {\n    padding: 6px 10px;\n    font-size: var(--jinks-file-manager-font-size-small);\n  }\n}\n\n/* Scrollbar Styling */\n.grid-container::-webkit-scrollbar {\n  width: 8px;\n}\n\n.grid-container::-webkit-scrollbar-track {\n  background: var(--jinks-file-manager-bg-scrollbar-track);\n}\n\n.grid-container::-webkit-scrollbar-thumb {\n  background: var(--jinks-file-manager-bg-scrollbar-thumb);\n  border-radius: 4px;\n}\n\n.grid-container::-webkit-scrollbar-thumb:hover {\n  background: var(--jinks-file-manager-bg-scrollbar-thumb-hover);\n}\n\n";
const MIME_TYPE_MAP = {
  // Images
  "image/jpeg": "image",
  "image/jpg": "image",
  "image/png": "image",
  "image/gif": "image",
  "image/svg+xml": "image",
  "image/webp": "image",
  "image/bmp": "image",
  "image/x-icon": "image",
  "image/vnd.microsoft.icon": "image",
  "image/tiff": "image",
  "image/tif": "image",
  // XML
  "application/xml": "xml",
  "text/xml": "xml",
  "application/xml-external-parsed-entity": "xml",
  "application/xml-dtd": "xml",
  // XQuery
  "application/xquery": "xquery",
  // JSON
  "application/json": "json",
  "text/json": "json",
  "application/json-patch+json": "json",
  "application/vnd.api+json": "json",
  // Javascript
  "application/javascript": "javascript",
  "text/javascript": "javascript",
  "application/ecmascript": "javascript",
  "text/ecmascript": "javascript",
  // CSS
  "text/css": "css",
  // HTML
  "text/html": "html",
  "application/xhtml+xml": "html",
  "application/xml+xhtml": "html",
  "application/xml-xhtml": "html",
  "application/xml-xhtml+xml": "html",
  "application/xml-xhtml+xml": "html"
};
const EXTENSION_MAP = {
  // Images
  "jpg": "image",
  "jpeg": "image",
  "png": "image",
  "gif": "image",
  "svg": "image",
  "webp": "image",
  "bmp": "image",
  "ico": "image",
  "tiff": "image",
  "tif": "image",
  // XML
  "xml": "xml",
  // JSON
  "json": "json",
  // CSS
  "css": "css",
  "sass": "css",
  "html": "html",
  "xquery": "xquery",
  "xql": "xquery",
  "xqm": "xquery",
  "xq": "xquery",
  "js": "javascript",
  "ts": "javascript"
};
function getFileType(item) {
  var _a, _b;
  if (item.type === "collection") return null;
  const mimeType = item.mime || item["mime-type"] || item.mimeType;
  if (mimeType) {
    const normalizedMimeType = mimeType.toLowerCase().split(";")[0].trim();
    if (MIME_TYPE_MAP[normalizedMimeType]) {
      return MIME_TYPE_MAP[normalizedMimeType];
    }
    if (normalizedMimeType.includes("xml")) {
      return "xml";
    }
    if (normalizedMimeType.includes("json")) {
      return "json";
    }
    if (normalizedMimeType.includes("css")) {
      return "css";
    }
    if (normalizedMimeType.startsWith("image/")) {
      return "image";
    }
  }
  const name = item.name || ((_a = item.path) == null ? void 0 : _a.split("/").pop()) || "";
  const ext = (_b = name.split(".").pop()) == null ? void 0 : _b.toLowerCase();
  if (ext && EXTENSION_MAP[ext]) {
    return EXTENSION_MAP[ext];
  }
  return null;
}
function isImageFile(item) {
  return getFileType(item) === "image";
}
class FileManager extends HTMLElement {
  constructor() {
    super();
    this.attachShadow({ mode: "open" });
    this.apiBase = "/exist/apps/jinks";
    this.root = "/db/apps/jinks";
    this.currentPath = this.root;
    this.items = [];
    this.selectedItems = /* @__PURE__ */ new Set();
    this.lastSelectedIndex = null;
    this.clipboard = null;
    this.clipboardMode = "copy";
    this.loadedRanges = [];
    this.pageSize = 100;
    this.loading = false;
    this.cache = /* @__PURE__ */ new Map();
    this.messageTimeout = null;
    this.handleClick = this.handleClick.bind(this);
    this.handleContextMenu = this.handleContextMenu.bind(this);
    this.handleKeyDown = this.handleKeyDown.bind(this);
    this.handlePaste = this.handlePaste.bind(this);
    this.handleDragEnter = this.handleDragEnter.bind(this);
    this.handleDragOver = this.handleDragOver.bind(this);
    this.handleDragLeave = this.handleDragLeave.bind(this);
    this.handleDrop = this.handleDrop.bind(this);
  }
  static get observedAttributes() {
    return ["api-base", "root"];
  }
  attributeChangedCallback(name, oldValue, newValue) {
    if (oldValue === newValue) return;
    switch (name) {
      case "api-base":
        this.apiBase = newValue || "/exist/apps/eXide";
        if (this.isConnected && this.shadowRoot && this.shadowRoot.querySelector(".grid-container")) {
          setTimeout(() => {
            this.loadCollection(this.currentPath);
          }, 0);
        }
        break;
      case "root":
        this.root = newValue;
        this.currentPath = this.root;
        this.cache.clear();
        this.loadedRanges = [];
        this.items = [];
        this.selectedItems.clear();
        if (this.root && this.root.trim() && this.isConnected && this.shadowRoot && this.shadowRoot.querySelector(".grid-container")) {
          setTimeout(() => {
            this.loadCollection(this.currentPath);
          }, 0);
        }
        break;
    }
  }
  connectedCallback() {
    this.render();
    if (this.root && this.root.trim()) {
      setTimeout(() => {
        this.loadCollection(this.currentPath);
      }, 0);
    }
    this.setupEventListeners();
  }
  disconnectedCallback() {
    this.removeEventListeners();
  }
  setupEventListeners() {
    this.shadowRoot.addEventListener("click", this.handleClick);
    this.shadowRoot.addEventListener("contextmenu", this.handleContextMenu);
    document.addEventListener("keydown", this.handleKeyDown);
    document.addEventListener("paste", this.handlePaste);
    const content = this.shadowRoot.querySelector(".content");
    if (content) {
      content.addEventListener("dragenter", this.handleDragEnter);
      content.addEventListener("dragover", this.handleDragOver);
      content.addEventListener("dragleave", this.handleDragLeave);
      content.addEventListener("drop", this.handleDrop);
    }
  }
  removeEventListeners() {
    this.shadowRoot.removeEventListener("click", this.handleClick);
    this.shadowRoot.removeEventListener("contextmenu", this.handleContextMenu);
    document.removeEventListener("keydown", this.handleKeyDown);
    document.removeEventListener("paste", this.handlePaste);
    const content = this.shadowRoot.querySelector(".content");
    if (content) {
      content.removeEventListener("dragenter", this.handleDragEnter);
      content.removeEventListener("dragover", this.handleDragOver);
      content.removeEventListener("dragleave", this.handleDragLeave);
      content.removeEventListener("drop", this.handleDrop);
    }
  }
  render() {
    this.shadowRoot.innerHTML = `
      <style>${stylesCss}</style>
      ${iconsSvg}
      <div class="file-manager">
        <div class="toolbar">
          <button class="btn-create-collection" title="Create collection">
            <svg width="16" height="16" fill="currentColor"><use href="#icon-folder-plus"></use></svg>
            New Collection
          </button>
          <button class="btn-upload" title="Upload files">
            <svg width="16" height="16" fill="currentColor"><use href="#icon-upload"></use></svg>
            Upload
          </button>
          <button class="btn-paste" disabled title="Paste (Cmd+V / Ctrl+V)">
            <svg width="16" height="16" fill="currentColor"><use href="#icon-clipboard"></use></svg>
            Paste
          </button>
          <div class="spacer"></div>
          <div class="clipboard-indicator" style="display: none;">
            <svg width="16" height="16" fill="currentColor"><use href="#icon-clipboard"></use></svg>
            <span class="clipboard-text"></span>
          </div>
        </div>
        <div class="breadcrumb"></div>
        <div class="content">
          <div class="grid-container"></div>
          <div class="loading" style="display: none;">
            <div class="spinner"></div>
            <span>Loading...</span>
          </div>
          <div class="load-more-container" style="display: none;">
            <button class="btn-load-more">Load More</button>
          </div>
          <div class="empty-state" style="display: none;">
            <p>This collection is empty</p>
          </div>
        </div>
        <div class="message-footer" style="display: none;">
          <span class="message-text"></span>
          <input type="text" class="message-input" style="display: none;" placeholder="Enter value...">
          <div class="message-actions">
            <button class="message-confirm-btn" style="display: none;">Confirm</button>
            <button class="message-cancel-btn" style="display: none;">Cancel</button>
            <button class="message-close" title="Close">×</button>
          </div>
        </div>
        <div class="context-menu" style="display: none;"></div>
        <input type="file" class="file-input" multiple style="display: none;">
      </div>
    `;
    const createCollectionBtn = this.shadowRoot.querySelector(".btn-create-collection");
    createCollectionBtn.addEventListener("click", () => this.performCreateCollection());
    const uploadBtn = this.shadowRoot.querySelector(".btn-upload");
    const fileInput = this.shadowRoot.querySelector(".file-input");
    uploadBtn.addEventListener("click", () => fileInput.click());
    fileInput.addEventListener("change", (e) => this.handleFileSelect(e));
    const pasteBtn = this.shadowRoot.querySelector(".btn-paste");
    pasteBtn.addEventListener("click", () => this.performPaste());
    this.updatePasteButton();
    const messageClose = this.shadowRoot.querySelector(".message-close");
    if (messageClose) {
      messageClose.addEventListener("click", () => this.hideMessage());
    }
  }
  // API Service Methods
  async fetchCollections(path, start = 0, end = this.pageSize) {
    const url = `${this.apiBase}/api/collections/${encodeURIComponent(path)}?start=${start}&end=${end}`;
    try {
      const response = await fetch(url);
      if (!response.ok) {
        const errorText = await response.text();
        console.error("HTTP error response:", errorText);
        throw new Error(`HTTP error! status: ${response.status} - ${errorText.substring(0, 100)}`);
      }
      const contentType = response.headers.get("content-type") || "";
      const text = await response.text();
      if (!text.trim().match(/^[\s]*[{\[]/)) {
        console.warn("Response does not appear to be JSON, content-type:", contentType);
        console.warn("Response text:", text.substring(0, 500));
        throw new Error(`Response is not valid JSON. Content-type: ${contentType}`);
      }
      let data;
      try {
        data = JSON.parse(text);
      } catch (parseError) {
        console.error("Failed to parse JSON:", parseError);
        console.error("Response text:", text.substring(0, 500));
        throw new Error(`Failed to parse JSON response: ${parseError.message}`);
      }
      return data;
    } catch (error) {
      console.error("Error fetching collections:", error);
      this.showError(`Failed to load collection: ${error.message}`);
      throw error;
    }
  }
  async uploadFile(collectionPath, file) {
    const formData = new FormData();
    formData.append("file[]", file);
    const url = new URL(`${this.apiBase}/api/upload`, window.location.origin);
    url.searchParams.append("collection", collectionPath);
    try {
      const response = await fetch(url.toString(), {
        method: "POST",
        body: formData
      });
      if (!response.ok) {
        const errorText = await response.text();
        console.error("Upload HTTP error:", response.status, errorText);
        throw new Error(`HTTP error! status: ${response.status} - ${errorText.substring(0, 100)}`);
      }
      const contentType = response.headers.get("content-type") || "";
      let result;
      if (contentType.includes("application/json")) {
        result = await response.json();
      } else {
        const text = await response.text();
        try {
          result = JSON.parse(text);
        } catch (e) {
          result = { success: true, message: text };
        }
      }
      return result;
    } catch (error) {
      console.error("Error uploading file:", error);
      this.showError(`Failed to upload file: ${error.message}`);
      throw error;
    }
  }
  async createCollection(collectionPath, name) {
    const url = `${this.apiBase}/api/collections/${encodeURIComponent(collectionPath)}?name=${encodeURIComponent(name)}`;
    try {
      const response = await fetch(url, {
        method: "POST"
      });
      if (!response.ok) {
        const errorText = await response.text();
        console.error("Create collection HTTP error:", response.status, errorText);
        throw new Error(`HTTP error! status: ${response.status} - ${errorText.substring(0, 100)}`);
      }
      const result = await response.json();
      if (result.status === "fail" && result.message) {
        console.error("Create collection failed, response:", result);
        throw new Error(result.message);
      }
      return result;
    } catch (error) {
      console.error("Error creating collection:", error);
      this.showError(`Failed to create collection: ${error.message}`);
      throw error;
    }
  }
  async deleteItem(path) {
    return this.deleteItems([path]);
  }
  async deleteItems(paths) {
    const collectionPath = this.currentPath;
    if (!Array.isArray(paths)) {
      console.error("deleteItems: paths is not an array:", paths);
      paths = [paths];
    }
    const validPaths = paths.filter((p) => p != null && p !== "");
    if (validPaths.length === 0) {
      throw new Error("No valid paths to delete");
    }
    const removeParams = validPaths.map((path) => `remove=${encodeURIComponent(path)}`).join("&");
    const url = `${this.apiBase}/api/collections/${encodeURIComponent(collectionPath)}?${removeParams}`;
    try {
      const response = await fetch(url, {
        method: "DELETE"
      });
      if (!response.ok) {
        const errorText = await response.text();
        console.error("Delete HTTP error:", response.status, errorText);
        throw new Error(`HTTP error! status: ${response.status} - ${errorText.substring(0, 100)}`);
      }
      const contentType = response.headers.get("content-type") || "";
      let result;
      if (contentType.includes("application/json") || contentType.includes("text/javascript")) {
        const text = await response.text();
        try {
          result = JSON.parse(text);
        } catch (e) {
          result = { success: true, message: text };
        }
      } else {
        const text = await response.text();
        result = { success: true, message: text };
      }
      if (result.status === "fail" && result.message) {
        console.error("Delete failed, response:", result);
        throw new Error(result.message);
      }
      return result;
    } catch (error) {
      console.error("Error deleting items:", error);
      this.showError(`Failed to delete item(s): ${error.message}`);
      throw error;
    }
  }
  async renameItem(oldPath, newName) {
    const collectionPath = this.currentPath;
    let resourceName;
    if (oldPath.includes("/")) {
      resourceName = oldPath.split("/").filter((p) => p).pop();
    } else {
      resourceName = oldPath;
    }
    const url = `${this.apiBase}/api/collections/${encodeURIComponent(collectionPath)}/resources/${encodeURIComponent(resourceName)}`;
    try {
      const response = await fetch(url, {
        method: "PATCH",
        headers: {
          "Content-Type": "application/json"
        },
        body: JSON.stringify({
          name: newName
        })
      });
      if (!response.ok) {
        const errorText = await response.text();
        console.error("Rename HTTP error:", response.status, errorText);
        throw new Error(`HTTP error! status: ${response.status} - ${errorText.substring(0, 100)}`);
      }
      const result = await response.json();
      return result;
    } catch (error) {
      console.error("Error renaming item:", error);
      this.showError(`Failed to rename item: ${error.message}`);
      throw error;
    }
  }
  async copyItem(sourcePath, targetCollection) {
    return this.copyItems([sourcePath], targetCollection);
  }
  async copyItems(sourcePaths, targetCollection) {
    if (!Array.isArray(sourcePaths)) {
      sourcePaths = [sourcePaths];
    }
    const validPaths = sourcePaths.filter((p) => p != null && p !== "");
    if (validPaths.length === 0) {
      throw new Error("No valid paths to copy");
    }
    let sourceCollection = this.currentPath;
    const firstPath = validPaths[0];
    if (firstPath.startsWith("/")) {
      const parts = firstPath.split("/").filter((p) => p);
      parts.pop();
      if (parts.length > 0) {
        sourceCollection = "/" + parts.join("/");
      }
    }
    const url = `${this.apiBase}/api/collections/${encodeURIComponent(sourceCollection)}/copy`;
    try {
      const response = await fetch(url, {
        method: "POST",
        headers: {
          "Content-Type": "application/json"
        },
        body: JSON.stringify({
          target: targetCollection,
          sources: validPaths
        })
      });
      if (!response.ok) {
        const errorText = await response.text();
        console.error("Copy HTTP error:", response.status, errorText);
        throw new Error(`HTTP error! status: ${response.status} - ${errorText.substring(0, 100)}`);
      }
      const result = await response.json();
      return result;
    } catch (error) {
      console.error("Error copying items:", error);
      this.showError(`Failed to copy item(s): ${error.message}`);
      throw error;
    }
  }
  async moveItems(sourcePaths, targetCollection) {
    if (!Array.isArray(sourcePaths)) {
      sourcePaths = [sourcePaths];
    }
    const validPaths = sourcePaths.filter((p) => p != null && p !== "");
    if (validPaths.length === 0) {
      throw new Error("No valid paths to move");
    }
    let sourceCollection = this.currentPath;
    const firstPath = validPaths[0];
    if (firstPath.startsWith("/")) {
      const parts = firstPath.split("/").filter((p) => p);
      parts.pop();
      if (parts.length > 0) {
        sourceCollection = "/" + parts.join("/");
      }
    }
    const url = `${this.apiBase}/api/collections/${encodeURIComponent(sourceCollection)}/move`;
    try {
      const response = await fetch(url, {
        method: "POST",
        headers: {
          "Content-Type": "application/json"
        },
        body: JSON.stringify({
          target: targetCollection,
          sources: validPaths
        })
      });
      if (!response.ok) {
        const errorText = await response.text();
        console.error("Move HTTP error:", response.status, errorText);
        throw new Error(`HTTP error! status: ${response.status} - ${errorText.substring(0, 100)}`);
      }
      const result = await response.json();
      return result;
    } catch (error) {
      console.error("Error moving items:", error);
      this.showError(`Failed to move item(s): ${error.message}`);
      throw error;
    }
  }
  // State Management
  async loadCollection(path, append = false) {
    if (this.loading) return;
    if (!this.shadowRoot || !this.shadowRoot.querySelector(".grid-container")) {
      console.warn("Shadow root not ready, deferring loadCollection");
      return;
    }
    this.loading = true;
    this.currentPath = path;
    const loadingEl = this.shadowRoot.querySelector(".loading");
    const gridContainer = this.shadowRoot.querySelector(".grid-container");
    const emptyState = this.shadowRoot.querySelector(".empty-state");
    const loadMoreContainer = this.shadowRoot.querySelector(".load-more-container");
    if (!gridContainer || !loadingEl || !emptyState || !loadMoreContainer) {
      console.error("Required DOM elements not found in shadow root");
      this.loading = false;
      return;
    }
    if (!append) {
      this.items = [];
      this.loadedRanges = [];
      this.selectedItems.clear();
      this.lastSelectedIndex = null;
      gridContainer.innerHTML = "";
      emptyState.style.display = "none";
      loadMoreContainer.style.display = "none";
      this.cache.delete(path);
    }
    loadingEl.style.display = "flex";
    try {
      const start = append ? this.items.length : 0;
      const end = start + this.pageSize;
      const rangeKey = `${start}-${end}`;
      if (append && this.loadedRanges.includes(rangeKey)) {
        this.loading = false;
        loadingEl.style.display = "none";
        return;
      }
      const data = await this.fetchCollections(path, start, end);
      let newItems = [];
      if (data && data.items) {
        newItems = Array.isArray(data.items) ? data.items : [data.items];
      } else if (data && data.collection) {
        newItems = Array.isArray(data.collection) ? data.collection : [data.collection];
      } else if (data && (data.resources || data.collections)) {
        const resources = data.resources ? Array.isArray(data.resources) ? data.resources : [data.resources] : [];
        const collections = data.collections ? Array.isArray(data.collections) ? data.collections : [data.collections] : [];
        newItems = [
          ...collections.map((c) => ({ ...c, type: "collection" })),
          ...resources.map((r) => ({ ...r, type: "resource" }))
        ];
      } else if (Array.isArray(data)) {
        newItems = data;
      } else if (data && typeof data === "object") {
        const arrayKeys = Object.keys(data).filter((key) => Array.isArray(data[key]));
        if (arrayKeys.length > 0) {
          newItems = data[arrayKeys[0]];
        } else {
          const items = [];
          Object.keys(data).forEach((key) => {
            if (data[key] && typeof data[key] === "object") {
              items.push({ ...data[key], name: data[key].name || key });
            }
          });
          newItems = items;
        }
      }
      newItems = newItems.map((item, index) => {
        if (!item.name) {
          if (item.key) {
            const keyParts = item.key.split("/").filter((p) => p);
            item.name = keyParts.length > 0 ? keyParts[keyParts.length - 1] : item.key;
          } else if (item.path) {
            const pathParts = item.path.split("/").filter((p) => p);
            item.name = pathParts.length > 0 ? pathParts[pathParts.length - 1] : item.path;
          }
        }
        if (item.name === "..") {
          let parentPath;
          if (item.key && item.key !== path) {
            parentPath = item.key;
          } else {
            const parts = path.split("/").filter((p) => p);
            parts.pop();
            parentPath = parts.length > 0 ? "/" + parts.join("/") : "/";
          }
          if (parentPath.startsWith(this.root)) {
            item.path = parentPath;
            item.name = "..";
          } else {
            return null;
          }
        } else {
          if (!item.path) {
            if (item.key) {
              item.path = item.key;
            } else if (item.name) {
              item.path = path.endsWith("/") ? path + item.name : path + "/" + item.name;
            }
          }
        }
        if (item.isCollection !== void 0) {
          item.type = item.isCollection ? "collection" : "resource";
        }
        if (!item.type) {
          if (item.name === "..") {
            item.type = "collection";
          } else {
            item.type = item.name && !item.name.includes(".") ? "collection" : "resource";
          }
        }
        if (!item.name || !item.path) {
          return null;
        }
        return item;
      }).filter((item) => item != null);
      newItems = newItems.filter((item) => {
        if (item.path && !item.path.startsWith(this.root)) {
          return false;
        }
        return true;
      });
      if (append) {
        this.items = [...this.items, ...newItems];
      } else {
        this.items = newItems;
      }
      if (newItems.length > 0 || !append) {
        this.loadedRanges.push(rangeKey);
      }
      this.renderGrid();
      this.updateBreadcrumb();
      this.updateLoadMoreButton();
      if (this.items.length === 0) {
        emptyState.style.display = "block";
      } else {
        emptyState.style.display = "none";
      }
      this.cache.set(path, { items: this.items, timestamp: Date.now() });
    } catch (error) {
      console.error("Error loading collection:", error);
      this.showError(`Failed to load collection: ${error.message}`);
      emptyState.style.display = "block";
      const emptyStateEl = this.shadowRoot.querySelector(".empty-state");
      if (emptyStateEl) {
        emptyStateEl.innerHTML = `<p style="color: #d32f2f;">Error: ${error.message}</p><p style="font-size: 12px; color: #666; margin-top: 8px;">Check the browser console for details.</p>`;
      }
    } finally {
      this.loading = false;
      loadingEl.style.display = "none";
    }
  }
  // Helper method to get REST API URL for an image
  getImageUrl(itemPath) {
    const restBase = this.apiBase.replace("/apps/jinks", "/rest");
    return `${restBase}${itemPath}`;
  }
  // Helper method to get REST API URL for a file
  getFileUrl(itemPath) {
    const restBase = this.apiBase.replace("/apps/jinks", "/rest");
    return `${restBase}${itemPath}`;
  }
  // Fetch file content from REST API
  async fetchFileContent(itemPath) {
    const url = this.getFileUrl(itemPath);
    try {
      const response = await fetch(url);
      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }
      return await response.blob();
    } catch (error) {
      console.error("Error fetching file:", error);
      throw error;
    }
  }
  // Download selected files to local directory using File System Access API
  async downloadFilesToDirectory() {
    if (!("showDirectoryPicker" in window)) {
      this.showError("File System Access API is not supported in this browser. Please use a modern browser like Chrome, Edge, or Opera.");
      return;
    }
    const selectedPaths = Array.from(this.selectedItems);
    const selectedItems = selectedPaths.map((path) => this.items.find((i) => (i.path || i.name) === path)).filter((item) => item != null && item.type === "resource");
    if (selectedItems.length === 0) {
      this.showError("Please select at least one file to download.");
      return;
    }
    try {
      const directoryHandle = await window.showDirectoryPicker({
        mode: "readwrite"
      });
      this.showMessage(`Downloading ${selectedItems.length} file(s)...`, "info");
      let successCount = 0;
      let errorCount = 0;
      for (const item of selectedItems) {
        try {
          const itemPath = item.path || item.name;
          const fileName = item.name || itemPath.split("/").pop();
          const blob = await this.fetchFileContent(itemPath);
          const fileHandle = await directoryHandle.getFileHandle(fileName, { create: true });
          const writable = await fileHandle.createWritable();
          await writable.write(blob);
          await writable.close();
          successCount++;
        } catch (error) {
          console.error(`Error downloading ${item.name}:`, error);
          errorCount++;
        }
      }
      if (errorCount === 0) {
        this.showMessage(`Successfully downloaded ${successCount} file(s) to selected directory.`);
      } else {
        this.showError(`Downloaded ${successCount} file(s), ${errorCount} failed.`);
      }
    } catch (error) {
      if (error.name !== "AbortError") {
        console.error("Error downloading files:", error);
        this.showError(`Failed to download files: ${error.message}`);
      }
    }
  }
  // UI Rendering
  renderGrid() {
    const gridContainer = this.shadowRoot.querySelector(".grid-container");
    gridContainer.innerHTML = "";
    this.items.forEach((item, index) => {
      var _a;
      const gridItem = document.createElement("div");
      gridItem.className = `grid-item ${item.type === "collection" ? "folder" : "file"}`;
      gridItem.dataset.path = item.path || item.name;
      gridItem.dataset.index = index;
      if (this.selectedItems.has(item.path || item.name)) {
        gridItem.classList.add("selected");
      }
      let name;
      if (item.name === "..") {
        name = "..";
      } else {
        name = item.name || ((_a = item.path) == null ? void 0 : _a.split("/").pop()) || "Unknown";
      }
      const itemPath = item.path || item.name;
      const isImage = isImageFile(item);
      let icon;
      if (item.type === "collection") {
        icon = `<svg width="48" height="48" fill="currentColor"><use href="#icon-folder"></use></svg>`;
      } else if (isImage) {
        const imageUrl = this.getImageUrl(itemPath);
        icon = `<img src="${imageUrl}" alt="${name}" class="thumbnail-image" data-path="${itemPath}">
                <svg width="48" height="48" fill="currentColor" class="thumbnail-fallback" style="display: none;"><use href="#icon-file"></use></svg>`;
      } else {
        const fileType = getFileType(item);
        if (fileType) {
          icon = `<svg width="48" height="48" fill="currentColor"><use href="#icon-filetype-${fileType}"></use></svg>`;
        } else {
          icon = `<svg width="48" height="48" fill="currentColor"><use href="#icon-file"></use></svg>`;
        }
      }
      gridItem.innerHTML = `
        <div class="item-icon">${icon}</div>
        <div class="item-name" title="${name}">${name}</div>
      `;
      if (isImage) {
        const img = gridItem.querySelector(".thumbnail-image");
        const fallback = gridItem.querySelector(".thumbnail-fallback");
        if (img && fallback) {
          img.addEventListener("error", () => {
            img.style.display = "none";
            fallback.style.display = "block";
          });
        }
      }
      gridContainer.appendChild(gridItem);
    });
  }
  updateBreadcrumb() {
    const breadcrumb = this.shadowRoot.querySelector(".breadcrumb");
    if (!breadcrumb) return;
    const currentParts = this.currentPath.split("/").filter((p) => p);
    const rootParts = this.root.split("/").filter((p) => p);
    breadcrumb.innerHTML = "";
    const homeBtn = document.createElement("button");
    homeBtn.className = "breadcrumb-item home";
    homeBtn.innerHTML = '<svg width="16" height="16" fill="currentColor"><use href="#icon-home"></use></svg>';
    homeBtn.title = "Root: " + this.root;
    if (this.currentPath !== this.root) {
      homeBtn.addEventListener("click", (e) => {
        e.preventDefault();
        e.stopPropagation();
        this.navigateTo(this.root);
      });
    } else {
      homeBtn.classList.add("current");
      homeBtn.disabled = true;
    }
    breadcrumb.appendChild(homeBtn);
    let currentPath = this.root;
    currentParts.slice(rootParts.length).forEach((part, index) => {
      currentPath += "/" + part;
      const segmentPath = currentPath;
      const separator = document.createElement("span");
      separator.className = "breadcrumb-separator";
      separator.textContent = "/";
      breadcrumb.appendChild(separator);
      const segment = document.createElement("button");
      segment.className = "breadcrumb-item";
      segment.textContent = part;
      segment.title = segmentPath;
      const relativeIndex = rootParts.length + index;
      if (relativeIndex === currentParts.length - 1) {
        segment.classList.add("current");
        segment.disabled = true;
      } else {
        segment.addEventListener("click", (e) => {
          e.preventDefault();
          e.stopPropagation();
          this.navigateTo(segmentPath);
        });
      }
      breadcrumb.appendChild(segment);
    });
  }
  updateLoadMoreButton() {
    const loadMoreContainer = this.shadowRoot.querySelector(".load-more-container");
    const loadMoreBtn = loadMoreContainer.querySelector(".btn-load-more");
    if (this.items.length > 0 && this.items.length % this.pageSize === 0) {
      loadMoreContainer.style.display = "block";
      loadMoreBtn.onclick = () => this.loadCollection(this.currentPath, true);
    } else {
      loadMoreContainer.style.display = "none";
    }
  }
  updatePasteButton() {
    const pasteBtn = this.shadowRoot.querySelector(".btn-paste");
    const clipboardIndicator = this.shadowRoot.querySelector(".clipboard-indicator");
    const clipboardText = this.shadowRoot.querySelector(".clipboard-text");
    const clipboardItems = this.clipboard ? Array.isArray(this.clipboard) ? this.clipboard : [this.clipboard] : [];
    if (clipboardItems.length > 0) {
      pasteBtn.disabled = false;
      clipboardIndicator.style.display = "flex";
      const count = clipboardItems.length;
      const action = this.clipboardMode === "cut" ? "Cut" : "Copied";
      if (count === 1) {
        const name = clipboardItems[0].name || clipboardItems[0].path.split("/").pop();
        clipboardText.textContent = `${action}: ${name}`;
      } else {
        clipboardText.textContent = `${action}: ${count} items`;
      }
    } else {
      pasteBtn.disabled = true;
      clipboardIndicator.style.display = "none";
    }
  }
  // Event Handlers
  handleClick(e) {
    const gridItem = e.target.closest(".grid-item");
    if (!gridItem) {
      if (!e.ctrlKey && !e.metaKey && !e.shiftKey) {
        this.selectedItems.clear();
        this.lastSelectedIndex = null;
        this.renderGrid();
      }
      this.hideContextMenu();
      return;
    }
    const path = gridItem.dataset.path;
    const item = this.items.find((i) => (i.path || i.name) === path);
    if (!item) return;
    if (item.type === "collection" || gridItem.classList.contains("folder")) {
      if (e.shiftKey || e.ctrlKey || e.metaKey) {
        this.toggleSelection(path, e);
      } else if (e.detail === 2 || e.detail === 1 && e.target.closest(".item-icon")) {
        this.navigateTo(path);
      } else {
        if (e.detail === 1) {
          this.toggleSelection(path, e);
        }
      }
    } else {
      if (e.detail === 2) {
        const fileType = getFileType(item);
        const editableTypes = ["xquery", "javascript", "css", "xml", "json", "html"];
        if (fileType && editableTypes.includes(fileType)) {
          this.openInExide(path);
          return;
        }
      }
      this.toggleSelection(path, e);
    }
  }
  handleContextMenu(e) {
    e.preventDefault();
    const gridItem = e.target.closest(".grid-item");
    if (!gridItem) {
      this.hideContextMenu();
      return;
    }
    const path = gridItem.dataset.path;
    const item = this.items.find((i) => (i.path || i.name) === path);
    if (!item) return;
    if (!this.selectedItems.has(path) && !e.ctrlKey && !e.metaKey && !e.shiftKey) {
      this.selectedItems.clear();
      this.lastSelectedIndex = null;
      this.selectedItems.add(path);
      const clickedIndex = this.items.findIndex((i) => (i.path || i.name) === path);
      this.lastSelectedIndex = clickedIndex >= 0 ? clickedIndex : null;
      this.renderGrid();
    }
    this.showContextMenu(e.clientX, e.clientY, item);
  }
  handleKeyDown(e) {
    const isMac = navigator.platform.toUpperCase().indexOf("MAC") >= 0 || navigator.userAgent.toUpperCase().indexOf("MAC") >= 0;
    if (this.selectedItems.size > 0) {
      if (e.key === "Delete" || isMac && e.key === "Backspace") {
        e.preventDefault();
        this.performDelete();
        return;
      }
    }
    if (e.key === "F2" && this.selectedItems.size === 1) {
      e.preventDefault();
      const path = Array.from(this.selectedItems)[0];
      this.performRename(path);
      return;
    }
    const hasModifier = isMac ? e.metaKey && !e.ctrlKey : e.ctrlKey && !e.metaKey;
    if (!hasModifier || e.shiftKey || e.altKey) {
      return;
    }
    const key = e.key.toLowerCase();
    if (key === "c") {
      if (this.selectedItems.size === 0 && !this.shadowRoot.contains(document.activeElement)) {
        return;
      }
      e.preventDefault();
      this.performCopy();
      return;
    }
    if (key === "x") {
      if (this.selectedItems.size === 0 && !this.shadowRoot.contains(document.activeElement)) {
        return;
      }
      e.preventDefault();
      this.performCut();
      return;
    }
    if (key === "v") {
      if (this.clipboard) {
        e.preventDefault();
        this.performPaste();
        return;
      }
    }
  }
  handlePaste(e) {
    if (!this.clipboard) {
      return;
    }
    if (!this.shadowRoot.contains(document.activeElement) && this.selectedItems.size === 0) ;
    e.preventDefault();
    this.performPaste();
  }
  handleFileSelect(e) {
    const files = Array.from(e.target.files);
    if (files.length === 0) return;
    this.uploadFiles(files);
    e.target.value = "";
  }
  // Drag and drop handlers
  handleDragEnter(e) {
    e.preventDefault();
    e.stopPropagation();
    if (e.dataTransfer.types.includes("Files")) {
      const content = this.shadowRoot.querySelector(".content");
      if (content) {
        content.classList.add("drag-over");
      }
    }
  }
  handleDragOver(e) {
    e.preventDefault();
    e.stopPropagation();
    if (e.dataTransfer.types.includes("Files")) {
      e.dataTransfer.dropEffect = "copy";
    }
  }
  handleDragLeave(e) {
    e.preventDefault();
    e.stopPropagation();
    const content = this.shadowRoot.querySelector(".content");
    if (content && !content.contains(e.relatedTarget)) {
      content.classList.remove("drag-over");
    }
  }
  handleDrop(e) {
    e.preventDefault();
    e.stopPropagation();
    const content = this.shadowRoot.querySelector(".content");
    if (content) {
      content.classList.remove("drag-over");
    }
    const files = Array.from(e.dataTransfer.files);
    if (files.length === 0) return;
    this.uploadFiles(files);
  }
  // Operations
  openInExide(path) {
    const exide = window.open("", "eXide");
    if (exide && !exide.closed) {
      const app = exide.eXide;
      if (app) {
        exide.eXide.app.findDocument(path);
        exide.focus();
      } else {
        window.eXide_onload = function() {
          exide.eXide.app.findDocument(path);
        };
        const base = this.apiBase.replace("/apps/jinks", "/apps/eXide");
        exide.location = `${base}/index.html`;
      }
    }
  }
  toggleSelection(path, e = null) {
    const clickedIndex = this.items.findIndex((i) => (i.path || i.name) === path);
    if (this.selectedItems.has(path)) {
      if (e && (e.ctrlKey || e.metaKey)) {
        this.selectedItems.delete(path);
        if (this.lastSelectedIndex === clickedIndex) {
          const selectedIndices = this.items.map((item, idx) => this.selectedItems.has(item.path || item.name) ? idx : -1).filter((idx) => idx !== -1);
          this.lastSelectedIndex = selectedIndices.length > 0 ? selectedIndices[selectedIndices.length - 1] : null;
        }
      } else {
        return;
      }
    } else {
      if (e && e.shiftKey && this.lastSelectedIndex !== null) {
        const startIndex = Math.min(this.lastSelectedIndex, clickedIndex);
        const endIndex = Math.max(this.lastSelectedIndex, clickedIndex);
        for (let i = startIndex; i <= endIndex; i++) {
          if (i >= 0 && i < this.items.length) {
            const item = this.items[i];
            const itemPath = item.path || item.name;
            this.selectedItems.add(itemPath);
          }
        }
        this.lastSelectedIndex = clickedIndex;
      } else {
        if (!e || !e.shiftKey && !e.ctrlKey && !e.metaKey) {
          this.selectedItems.clear();
        }
        this.selectedItems.add(path);
        this.lastSelectedIndex = clickedIndex;
      }
    }
    this.renderGrid();
  }
  navigateTo(path) {
    const normalizePath = (p) => {
      if (!p) return "";
      p = p.trim();
      if (!p.startsWith("/")) p = "/" + p;
      if (p.length > 1 && p.endsWith("/")) {
        p = p.slice(0, -1);
      }
      return p;
    };
    const normalizedPath = normalizePath(path);
    const normalizedCurrent = normalizePath(this.currentPath);
    const normalizedRoot = normalizePath(this.root);
    if (normalizedPath && !normalizedPath.startsWith(normalizedRoot)) {
      console.warn("Navigation blocked: path is outside configured root", {
        path: normalizedPath,
        root: normalizedRoot
      });
      this.showError("Cannot navigate outside the configured root collection");
      return;
    }
    if (!path || normalizedPath === normalizedCurrent) {
      return;
    }
    this.loadCollection(normalizedPath);
  }
  async uploadFiles(files) {
    const loadingEl = this.shadowRoot.querySelector(".loading");
    if (!loadingEl) {
      console.error("Loading element not found");
      return;
    }
    loadingEl.style.display = "flex";
    try {
      const uploadPath = this.currentPath;
      for (const file of files) {
        await this.uploadFile(uploadPath, file);
      }
      this.cache.delete(uploadPath);
      this.loadedRanges = [];
      this.items = [];
      await new Promise((resolve) => setTimeout(resolve, 300));
      await this.loadCollection(uploadPath, false);
      this.showMessage(`Successfully uploaded ${files.length} file(s)`);
    } catch (error) {
      console.error("Upload error:", error);
      this.showError(`Failed to upload file(s): ${error.message}`);
    } finally {
      loadingEl.style.display = "none";
    }
  }
  performCopy() {
    if (this.selectedItems.size === 0) return;
    const selectedPaths = Array.from(this.selectedItems);
    const selectedItems = selectedPaths.map((path) => this.items.find((i) => (i.path || i.name) === path)).filter((item) => item != null);
    if (selectedItems.length === 0) return;
    this.clipboard = selectedItems.map((item) => {
      var _a;
      return {
        path: item.path || item.name,
        type: item.type || (((_a = item.name) == null ? void 0 : _a.endsWith("/")) ? "collection" : "resource"),
        name: item.name || (item.path || item.name).split("/").pop()
      };
    });
    this.clipboardMode = "copy";
    this.updatePasteButton();
    const count = this.clipboard.length;
    const names = this.clipboard.map((c) => c.name).join(", ");
    this.showMessage(`Copied ${count} item(s): ${names.substring(0, 50)}${names.length > 50 ? "..." : ""}`);
  }
  performCut() {
    if (this.selectedItems.size === 0) return;
    const selectedPaths = Array.from(this.selectedItems);
    const selectedItems = selectedPaths.map((path) => this.items.find((i) => (i.path || i.name) === path)).filter((item) => item != null);
    if (selectedItems.length === 0) return;
    this.clipboard = selectedItems.map((item) => {
      var _a;
      return {
        path: item.path || item.name,
        type: item.type || (((_a = item.name) == null ? void 0 : _a.endsWith("/")) ? "collection" : "resource"),
        name: item.name || (item.path || item.name).split("/").pop()
      };
    });
    this.clipboardMode = "cut";
    this.updatePasteButton();
    const count = this.clipboard.length;
    const names = this.clipboard.map((c) => c.name).join(", ");
    this.showMessage(`Cut ${count} item(s): ${names.substring(0, 50)}${names.length > 50 ? "..." : ""}`);
  }
  async copyPathToClipboard(fullPath) {
    let relativePath = fullPath;
    if (this.root && this.root.trim()) {
      const normalizedRoot = this.root.trim();
      const normalizedPath = fullPath.trim();
      if (normalizedPath.startsWith(normalizedRoot)) {
        relativePath = normalizedPath.substring(normalizedRoot.length);
        if (relativePath.startsWith("/")) {
          relativePath = relativePath.substring(1);
        }
      }
    }
    try {
      await navigator.clipboard.writeText(relativePath);
      this.showMessage(`Copied path to clipboard: ${relativePath}`);
    } catch (error) {
      console.error("Failed to copy to clipboard:", error);
      const textArea = document.createElement("textarea");
      textArea.value = relativePath;
      textArea.style.position = "fixed";
      textArea.style.left = "-999999px";
      document.body.appendChild(textArea);
      textArea.select();
      try {
        document.execCommand("copy");
        this.showMessage(`Copied path to clipboard: ${relativePath}`);
      } catch (err) {
        this.showError(`Failed to copy path: ${err.message}`);
      } finally {
        document.body.removeChild(textArea);
      }
    }
  }
  async performPaste() {
    if (!this.clipboard || this.clipboard.length === 0) return;
    try {
      const targetCollection = this.currentPath;
      const itemsToPaste = Array.isArray(this.clipboard) ? this.clipboard : [this.clipboard];
      const pathsToPaste = itemsToPaste.map((item) => item.path);
      if (this.clipboardMode === "cut") {
        await this.moveItems(pathsToPaste, targetCollection);
      } else {
        await this.copyItems(pathsToPaste, targetCollection);
      }
      this.cache.delete(targetCollection);
      this.loadedRanges = [];
      this.items = [];
      await this.loadCollection(targetCollection, false);
      const count = itemsToPaste.length;
      const names = itemsToPaste.map((item) => item.name).join(", ");
      const action = this.clipboardMode === "cut" ? "Moved" : "Pasted";
      this.showMessage(`${action} ${count} item(s): ${names.substring(0, 50)}${names.length > 50 ? "..." : ""}`);
      this.clipboard = null;
      this.clipboardMode = "copy";
      this.updatePasteButton();
    } catch (error) {
      console.error("Paste error:", error);
      this.showError(`Failed to paste item(s): ${error.message}`);
    }
  }
  async performDelete() {
    if (this.selectedItems.size === 0) return;
    const itemsToDelete = Array.from(this.selectedItems);
    const confirmMessage = `Are you sure you want to delete ${itemsToDelete.length} item(s)?`;
    const confirmed = await this.showConfirmation(confirmMessage, "error");
    if (!confirmed) return;
    const loadingEl = this.shadowRoot.querySelector(".loading");
    loadingEl.style.display = "flex";
    try {
      const deletePath = this.currentPath;
      await this.deleteItems(itemsToDelete);
      this.cache.delete(deletePath);
      this.loadedRanges = [];
      this.items = [];
      await this.loadCollection(deletePath, false);
      this.selectedItems.clear();
      this.lastSelectedIndex = null;
      this.showMessage(`Deleted ${itemsToDelete.length} item(s)`);
    } catch (error) {
      console.error("Delete error:", error);
      this.showError(`Failed to delete item(s): ${error.message}`);
    } finally {
      loadingEl.style.display = "none";
    }
  }
  async performCreateCollection() {
    const collectionName = await this.showPrompt("Enter collection name:", "", "info");
    if (!collectionName || !collectionName.trim()) return;
    const name = collectionName.trim();
    if (name.includes("/") || name.includes("\\")) {
      this.showError("Collection name cannot contain slashes");
      return;
    }
    try {
      await this.createCollection(this.currentPath, name);
      this.cache.delete(this.currentPath);
      this.loadedRanges = [];
      this.items = [];
      await this.loadCollection(this.currentPath, false);
      this.showMessage(`Collection "${name}" created successfully`);
    } catch (error) {
      console.error("Create collection error:", error);
    }
  }
  async performRename(path) {
    const item = this.items.find((i) => (i.path || i.name) === path);
    if (!item) return;
    const currentName = item.name || path.split("/").pop();
    const newName = await this.showPrompt("Enter new name:", currentName, "info");
    if (!newName || newName === currentName) return;
    try {
      await this.renameItem(path, newName);
      await this.loadCollection(this.currentPath);
      this.showMessage(`Renamed to: ${newName}`);
    } catch (error) {
      console.error("Rename error:", error);
    }
  }
  // Context Menu
  showContextMenu(x, y, item) {
    const contextMenu = this.shadowRoot.querySelector(".context-menu");
    contextMenu.style.display = "block";
    contextMenu.style.left = `${x}px`;
    contextMenu.style.top = `${y}px`;
    const path = item.path || item.name;
    this.selectedItems.has(path);
    const fileType = item.type === "resource" ? getFileType(item) : null;
    const editableTypes = ["xquery", "javascript", "css", "xml", "json", "html"];
    const isEditable = fileType && editableTypes.includes(fileType);
    const selectedPaths = Array.from(this.selectedItems);
    const hasSelectedFiles = selectedPaths.some((path2) => {
      const selectedItem = this.items.find((i) => (i.path || i.name) === path2);
      return selectedItem && selectedItem.type === "resource";
    });
    const isFile = item.type === "resource";
    const showDownload = (hasSelectedFiles || isFile) && "showDirectoryPicker" in window;
    contextMenu.innerHTML = `
      <div class="context-menu-item" data-action="open" ${item.type === "collection" ? "" : 'style="display: none;"'}>
        <svg width="16" height="16" fill="currentColor"><use href="#icon-folder"></use></svg>
        Open
      </div>
      <div class="context-menu-item" data-action="open-exide" ${isEditable ? "" : 'style="display: none;"'}>
        <svg width="16" height="16" fill="currentColor"><use href="#icon-edit"></use></svg>
        Open in eXide
      </div>
      <div class="context-menu-item" data-action="copy">
        <svg width="16" height="16" fill="currentColor"><use href="#icon-copy"></use></svg>
        Copy
      </div>
      <div class="context-menu-item" data-action="cut">
        <svg width="16" height="16" fill="currentColor"><use href="#icon-cut"></use></svg>
        Cut
      </div>
      <div class="context-menu-item" data-action="copy-path">
        <svg width="16" height="16" fill="currentColor"><use href="#icon-copy-path"></use></svg>
        Copy Relative Path
      </div>
      <div class="context-menu-item" data-action="paste" ${this.clipboard ? "" : 'style="display: none;"'}>
        <svg width="16" height="16" fill="currentColor"><use href="#icon-clipboard"></use></svg>
        Paste
      </div>
      <div class="context-menu-separator" ${showDownload ? "" : 'style="display: none;"'}></div>
      <div class="context-menu-item" data-action="download" ${showDownload ? "" : 'style="display: none;"'}>
        <svg width="16" height="16" fill="currentColor"><use href="#icon-download"></use></svg>
        Download to Folder
      </div>
      <div class="context-menu-separator"></div>
      <div class="context-menu-item" data-action="rename">
        <svg width="16" height="16" fill="currentColor"><use href="#icon-rename"></use></svg>
        Rename
      </div>
      <div class="context-menu-item" data-action="delete">
        <svg width="16" height="16" fill="currentColor"><use href="#icon-delete"></use></svg>
        Delete
      </div>
    `;
    contextMenu.querySelectorAll(".context-menu-item").forEach((item2) => {
      item2.addEventListener("click", (e) => {
        const action = item2.dataset.action;
        this.handleContextMenuAction(action, path, item2);
        this.hideContextMenu();
      });
    });
    setTimeout(() => {
      document.addEventListener("click", () => this.hideContextMenu(), { once: true });
    }, 0);
  }
  hideContextMenu() {
    const contextMenu = this.shadowRoot.querySelector(".context-menu");
    contextMenu.style.display = "none";
  }
  handleContextMenuAction(action, path, item) {
    switch (action) {
      case "open":
        this.navigateTo(path);
        break;
      case "open-exide":
        this.openInExide(path);
        break;
      case "download":
        this.downloadFilesToDirectory();
        break;
      case "copy":
        if (!this.selectedItems.has(path)) {
          this.selectedItems.clear();
          this.lastSelectedIndex = null;
          this.selectedItems.add(path);
          const clickedIndex = this.items.findIndex((i) => (i.path || i.name) === path);
          this.lastSelectedIndex = clickedIndex >= 0 ? clickedIndex : null;
          this.renderGrid();
        }
        this.performCopy();
        break;
      case "cut":
        if (!this.selectedItems.has(path)) {
          this.selectedItems.clear();
          this.lastSelectedIndex = null;
          this.selectedItems.add(path);
          const clickedIndex = this.items.findIndex((i) => (i.path || i.name) === path);
          this.lastSelectedIndex = clickedIndex >= 0 ? clickedIndex : null;
          this.renderGrid();
        }
        this.performCut();
        break;
      case "copy-path":
        this.copyPathToClipboard(path);
        break;
      case "paste":
        this.performPaste();
        break;
      case "rename":
        if (!this.selectedItems.has(path)) {
          this.selectedItems.clear();
          this.lastSelectedIndex = null;
          this.selectedItems.add(path);
          const clickedIndex = this.items.findIndex((i) => (i.path || i.name) === path);
          this.lastSelectedIndex = clickedIndex >= 0 ? clickedIndex : null;
          this.renderGrid();
        }
        this.performRename(path);
        break;
      case "delete":
        if (!this.selectedItems.has(path)) {
          this.selectedItems.clear();
          this.lastSelectedIndex = null;
          this.selectedItems.add(path);
          const clickedIndex = this.items.findIndex((i) => (i.path || i.name) === path);
          this.lastSelectedIndex = clickedIndex >= 0 ? clickedIndex : null;
          this.renderGrid();
        }
        this.performDelete();
        break;
    }
  }
  // Utility Methods
  showError(message) {
    this.showMessage(message, "error");
  }
  showMessage(message, type = "info") {
    const messageFooter = this.shadowRoot.querySelector(".message-footer");
    const messageText = this.shadowRoot.querySelector(".message-text");
    const messageInput = this.shadowRoot.querySelector(".message-input");
    const confirmBtn = this.shadowRoot.querySelector(".message-confirm-btn");
    const cancelBtn = this.shadowRoot.querySelector(".message-cancel-btn");
    const closeBtn = this.shadowRoot.querySelector(".message-close");
    if (!messageFooter || !messageText) {
      alert(message);
      return;
    }
    if (confirmBtn) confirmBtn.style.display = "none";
    if (cancelBtn) cancelBtn.style.display = "none";
    if (messageInput) messageInput.style.display = "none";
    if (closeBtn) closeBtn.style.display = "block";
    messageText.textContent = message;
    messageFooter.className = `message-footer message-${type}`;
    messageFooter.style.display = "flex";
    const hideDelay = type === "error" ? 1e4 : 5e3;
    if (this.messageTimeout) {
      clearTimeout(this.messageTimeout);
    }
    this.messageTimeout = setTimeout(() => {
      this.hideMessage();
    }, hideDelay);
  }
  showConfirmation(message, type = "info") {
    return new Promise((resolve) => {
      const messageFooter = this.shadowRoot.querySelector(".message-footer");
      const messageText = this.shadowRoot.querySelector(".message-text");
      const messageInput = this.shadowRoot.querySelector(".message-input");
      const confirmBtn = this.shadowRoot.querySelector(".message-confirm-btn");
      const cancelBtn = this.shadowRoot.querySelector(".message-cancel-btn");
      const closeBtn = this.shadowRoot.querySelector(".message-close");
      if (!messageFooter || !messageText || !confirmBtn || !cancelBtn) {
        const result = confirm(message);
        resolve(result);
        return;
      }
      if (messageInput) messageInput.style.display = "none";
      confirmBtn.style.display = "block";
      cancelBtn.style.display = "block";
      closeBtn.style.display = "none";
      messageText.textContent = message;
      messageFooter.className = `message-footer message-${type}`;
      messageFooter.style.display = "flex";
      if (this.messageTimeout) {
        clearTimeout(this.messageTimeout);
        this.messageTimeout = null;
      }
      const cleanup = () => {
        messageFooter.style.display = "none";
        confirmBtn.style.display = "none";
        cancelBtn.style.display = "none";
        closeBtn.style.display = "block";
        confirmBtn.onclick = null;
        cancelBtn.onclick = null;
      };
      confirmBtn.onclick = () => {
        cleanup();
        resolve(true);
      };
      cancelBtn.onclick = () => {
        cleanup();
        resolve(false);
      };
    });
  }
  showPrompt(message, defaultValue = "", type = "info") {
    return new Promise((resolve) => {
      const messageFooter = this.shadowRoot.querySelector(".message-footer");
      const messageText = this.shadowRoot.querySelector(".message-text");
      const messageInput = this.shadowRoot.querySelector(".message-input");
      const confirmBtn = this.shadowRoot.querySelector(".message-confirm-btn");
      const cancelBtn = this.shadowRoot.querySelector(".message-cancel-btn");
      const closeBtn = this.shadowRoot.querySelector(".message-close");
      if (!messageFooter || !messageText || !messageInput || !confirmBtn || !cancelBtn) {
        const result = prompt(message, defaultValue);
        resolve(result);
        return;
      }
      messageInput.style.display = "block";
      confirmBtn.style.display = "block";
      cancelBtn.style.display = "block";
      closeBtn.style.display = "none";
      messageText.textContent = message;
      messageInput.value = defaultValue;
      messageFooter.className = `message-footer message-${type}`;
      messageFooter.style.display = "flex";
      setTimeout(() => {
        messageInput.focus();
        messageInput.select();
      }, 0);
      if (this.messageTimeout) {
        clearTimeout(this.messageTimeout);
        this.messageTimeout = null;
      }
      const cleanup = () => {
        messageFooter.style.display = "none";
        messageInput.style.display = "none";
        confirmBtn.style.display = "none";
        cancelBtn.style.display = "none";
        closeBtn.style.display = "block";
        confirmBtn.onclick = null;
        cancelBtn.onclick = null;
        messageInput.onkeydown = null;
      };
      messageInput.onkeydown = (e) => {
        if (e.key === "Enter") {
          e.preventDefault();
          const value = messageInput.value.trim();
          cleanup();
          resolve(value || null);
        } else if (e.key === "Escape") {
          e.preventDefault();
          cleanup();
          resolve(null);
        }
      };
      confirmBtn.onclick = () => {
        const value = messageInput.value.trim();
        cleanup();
        resolve(value || null);
      };
      cancelBtn.onclick = () => {
        cleanup();
        resolve(null);
      };
    });
  }
  hideMessage() {
    const messageFooter = this.shadowRoot.querySelector(".message-footer");
    if (messageFooter) {
      messageFooter.style.display = "none";
    }
    if (this.messageTimeout) {
      clearTimeout(this.messageTimeout);
      this.messageTimeout = null;
    }
  }
}
customElements.define("jinks-file-manager", FileManager);
//# sourceMappingURL=jinks-file-manager.js.map
